/**************************************************************************
* DSemu: ARM7 portable interpretation core (arm7.c)                       *
* Released under the terms of the BSD Public Licence                      *
* Imran Nazar (tf@oopsilon.com), 2004                                     *
**************************************************************************/

#include <stdio.h>
#include "err.h"
#include "arm7.h"
#include "arm7func.h"
#include "cache.h"
#include "vtbl.h"
#include "dbgout.h"

u32 *dbuf;

int ARM7init(char *file, u32 *dbgbuf)
{
    char str[80];
    sprintf(str,"ARM7: ROM file: %s",file);
    logvt->append(str);

    MMUinit(file);
    dbuf=dbgbuf;

    ARM7reset();
    RETPASS("ARM7: Initialised.");
}

void ARM7fini()
{
    MMUfini();
    logvt->append("ARM7: Shutdown.");
}

void ARM7reset()
{
    int a;

    arm7reg.r0=0;  arm7reg.r1=0;  arm7reg.r2=0;  arm7reg.r3=0;
    arm7reg.r4=0;  arm7reg.r5=0;  arm7reg.r6=0;  arm7reg.r7=0;
    arm7reg.r8=0;  arm7reg.r9=0;  arm7reg.r10=0; arm7reg.r11=0;
    arm7reg.r12=0; arm7reg.r13=0x03007F00; arm7reg.r14=0; arm7reg.r15=0x08000000;

    arm7reg.r[0]=arm7reg.r0;   arm7reg.r[1]=arm7reg.r1;   arm7reg.r[2]=arm7reg.r2;   arm7reg.r[3]=arm7reg.r3;
    arm7reg.r[4]=arm7reg.r4;   arm7reg.r[5]=arm7reg.r5;   arm7reg.r[6]=arm7reg.r6;   arm7reg.r[7]=arm7reg.r7;
    arm7reg.r[8]=arm7reg.r8;   arm7reg.r[9]=arm7reg.r9;   arm7reg.r[10]=arm7reg.r10; arm7reg.r[11]=arm7reg.r11;
    arm7reg.r[12]=arm7reg.r12; arm7reg.r[13]=arm7reg.r13; arm7reg.r[14]=arm7reg.r14; arm7reg.r[15]=arm7reg.r15;

    arm7reg.cpsr=ARMS_M_SYS;
    arm7reg.clock=0;

    for(a=0;a<7;a++) arm7reg.spsr[a]=0;
    arm7reg.r8fiq=0;  arm7reg.r9fiq=0;  arm7reg.r10fiq=0; arm7reg.r11fiq=0;
    arm7reg.r12fiq=0; arm7reg.r13fiq=0; arm7reg.r14fiq=0;
    arm7reg.r13irq=0x03007FA0; arm7reg.r14irq=0;
    arm7reg.r13abt=0; arm7reg.r14abt=0;
    arm7reg.r13svc=0; arm7reg.r14svc=0;
    arm7reg.r13und=0; arm7reg.r14und=0;

    arm7reg.curmode=ARMMD_SYS;
    for(a=0;a<7;a++) arm7reg.flags[a]=0;
    arm7reg.flags[7]=1;
    MMUreset();
}

__int64 ARM7getClock() { return arm7reg.clock; }
void ARM7addClock(int clk) { arm7reg.clock+=clk; }

int ARM7exec()
{
    //int idx,cond;
    char str[512], opstr[128]; __int64 clk=arm7reg.clock,clkend;
    u32 r15=arm7reg.r[15];
    RAMWORD op;
        op=MMUrdS(0,r15);
	#ifdef ARM7DEBUG
//        sprintf(str,"%08X: Encountered opcode %08X (%s)", arm7reg.r[15], op.data, ARM7DASM(op.data));
//        logvt->append(str);
	#endif
	if(arm7reg.cpsr&ARMS_T)
	{
            switch(r15&2)
            {
                case 0:
		    arm7reg.curop=op.data&65535;
                    sprintf(opstr,"%s", Thumb7DASM(arm7reg.curop));
                    arm7reg.r[15]+=2;
		    thumb7ops[arm7reg.curop>>8].op();
                    arm7reg.clock+=thumb7ops[arm7reg.curop>>8].cyc; break;
                case 2:
		    arm7reg.curop=op.data>>16;
                    sprintf(opstr,"%s", Thumb7DASM(arm7reg.curop));
                    arm7reg.r[15]+=2;
		    thumb7ops[arm7reg.curop>>8].op();
                    arm7reg.clock+=thumb7ops[arm7reg.curop>>8].cyc; break;
            }
        } else {
            sprintf(opstr,"%s", ARM7DASM(op.data));
            arm7reg.r[15]+=4;
            if(ARM7cond[op.cond]())
            {
                arm7reg.curop=op.data;
                op.op();
                arm7reg.clock+=op.cyc;
            }
            else arm7reg.clock++;
        }
	clkend=arm7reg.clock;
	return clkend-clk;
}

int ARM7execfor(u32 count)
{
    //int idx,cond;
    char str[512], opstr[128]; __int64 clk=arm7reg.clock,clkend=clk+count;
    RAMWORD op;
    while(arm7reg.clock<clkend)
    {
        op=MMUrdS(0,arm7reg.r[15]);
	if(arm7reg.cpsr&ARMS_T)
	{
            switch(arm7reg.r[15]&2)
            {
                case 0:
		    arm7reg.curop=op.data&65535;
                    arm7reg.r[15]+=2;
		    thumb7ops[arm7reg.curop>>8].op();
                    arm7reg.clock+=thumb7ops[arm7reg.curop>>8].cyc; break;
                case 2:
		    arm7reg.curop=op.data>>16;
                    arm7reg.r[15]+=2;
		    thumb7ops[arm7reg.curop>>8].op();
                    arm7reg.clock+=thumb7ops[arm7reg.curop>>8].cyc; break;
            }
        } else {
            arm7reg.r[15]+=4;
            if(ARM7cond[op.cond]())
            {
                arm7reg.curop=op.data;
                op.op();
                arm7reg.clock+=op.cyc;
            }
        }
    }
    return 0;
}

void ARM7updateCPSR()
{
    if(arm7reg.flags[0]) arm7reg.cpsr|=ARMS_N;
    else arm7reg.cpsr&=(0xFFFFFFFF-ARMS_N);
    if(arm7reg.flags[1]) arm7reg.cpsr|=ARMS_Z;
    else arm7reg.cpsr&=(0xFFFFFFFF-ARMS_Z);
    if(arm7reg.flags[2]) arm7reg.cpsr|=ARMS_C;
    else arm7reg.cpsr&=(0xFFFFFFFF-ARMS_C);
    if(arm7reg.flags[3]) arm7reg.cpsr|=ARMS_V;
    else arm7reg.cpsr&=(0xFFFFFFFF-ARMS_V);
}

void ARM7splitCPSR()
{
    arm7reg.flags[0]=(arm7reg.cpsr>>31)&1;
    arm7reg.flags[1]=(arm7reg.cpsr>>30)&1;
    arm7reg.flags[2]=(arm7reg.cpsr>>29)&1;
    arm7reg.flags[3]=(arm7reg.cpsr>>28)&1;
}

void ARM7modesw(int mdin, int mdout)
{
    ARM7updateCPSR();
    arm7reg.r0=arm7reg.r[0]; arm7reg.r1=arm7reg.r[1];
    arm7reg.r2=arm7reg.r[2]; arm7reg.r3=arm7reg.r[3];
    arm7reg.r4=arm7reg.r[4]; arm7reg.r5=arm7reg.r[5];
    arm7reg.r6=arm7reg.r[6]; arm7reg.r7=arm7reg.r[7];
    arm7reg.r15=arm7reg.r[15];

    switch(mdin)
    {
    	case ARMMD_USR: case ARMMD_SYS:
    	    arm7reg.r8=arm7reg.r[8];   arm7reg.r9=arm7reg.r[9];
    	    arm7reg.r10=arm7reg.r[10]; arm7reg.r11=arm7reg.r[11];
    	    arm7reg.r12=arm7reg.r[12]; arm7reg.r13=arm7reg.r[13];
    	    arm7reg.r14=arm7reg.r[14];
    	    break;
    	case ARMMD_IRQ:
    	    arm7reg.r8=arm7reg.r[8];   arm7reg.r9=arm7reg.r[9];
    	    arm7reg.r10=arm7reg.r[10]; arm7reg.r11=arm7reg.r[11];
    	    arm7reg.r12=arm7reg.r[12];
            arm7reg.r13irq=arm7reg.r[13]; arm7reg.r14irq=arm7reg.r[14];
    	    break;
        case ARMMD_SVC:
    	    arm7reg.r8=arm7reg.r[8];   arm7reg.r9=arm7reg.r[9];
    	    arm7reg.r10=arm7reg.r[10]; arm7reg.r11=arm7reg.r[11];
    	    arm7reg.r12=arm7reg.r[12];
            arm7reg.r13svc=arm7reg.r[13]; arm7reg.r14svc=arm7reg.r[14];
    	    break;
        case ARMMD_ABT:
    	    arm7reg.r8=arm7reg.r[8];   arm7reg.r9=arm7reg.r[9];
    	    arm7reg.r10=arm7reg.r[10]; arm7reg.r11=arm7reg.r[11];
    	    arm7reg.r12=arm7reg.r[12];
            arm7reg.r13abt=arm7reg.r[13]; arm7reg.r14abt=arm7reg.r[14];
    	    break;
        case ARMMD_UND:
    	    arm7reg.r8=arm7reg.r[8];   arm7reg.r9=arm7reg.r[9];
    	    arm7reg.r10=arm7reg.r[10]; arm7reg.r11=arm7reg.r[11];
    	    arm7reg.r12=arm7reg.r[12];
            arm7reg.r13und=arm7reg.r[13]; arm7reg.r14und=arm7reg.r[14];
    	    break;
    	case ARMMD_FIQ:
            arm7reg.r8fiq=arm7reg.r[8];   arm7reg.r9fiq=arm7reg.r[9];
    	    arm7reg.r10fiq=arm7reg.r[10]; arm7reg.r11fiq=arm7reg.r[11];
    	    arm7reg.r12fiq=arm7reg.r[12]; arm7reg.r13fiq=arm7reg.r[13];
    	    arm7reg.r14fiq=arm7reg.r[14];
    	    break;
    }
    arm7reg.cpsr&=(0xFFFFFFFF-ARMS_M);
    switch(mdout)
    {
    	case ARMMD_USR:
    	    arm7reg.r[8]=arm7reg.r8;   arm7reg.r[9]=arm7reg.r9;
    	    arm7reg.r[10]=arm7reg.r10; arm7reg.r[11]=arm7reg.r11;
    	    arm7reg.r[12]=arm7reg.r12; arm7reg.r[13]=arm7reg.r13;
    	    arm7reg.r[14]=arm7reg.r14;
    	    break;
        case ARMMD_SYS:
    	    arm7reg.r[8]=arm7reg.r8;   arm7reg.r[9]=arm7reg.r9;
    	    arm7reg.r[10]=arm7reg.r10; arm7reg.r[11]=arm7reg.r11;
    	    arm7reg.r[12]=arm7reg.r12; arm7reg.r[13]=arm7reg.r13;
    	    arm7reg.r[14]=arm7reg.r14;
    	    break;
    	case ARMMD_IRQ:
    	    arm7reg.r[8]=arm7reg.r8;   arm7reg.r[9]=arm7reg.r9;
    	    arm7reg.r[10]=arm7reg.r10; arm7reg.r[11]=arm7reg.r11;
    	    arm7reg.r[12]=arm7reg.r12;
            arm7reg.r[13]=arm7reg.r13irq; arm7reg.r[14]=arm7reg.r14irq;
    	    break;
        case ARMMD_SVC:
    	    arm7reg.r[8]=arm7reg.r8;   arm7reg.r[9]=arm7reg.r9;
    	    arm7reg.r[10]=arm7reg.r10; arm7reg.r[11]=arm7reg.r11;
    	    arm7reg.r[12]=arm7reg.r12;
            arm7reg.r[13]=arm7reg.r13svc; arm7reg.r[14]=arm7reg.r14svc;
    	    break;
        case ARMMD_ABT:
    	    arm7reg.r[8]=arm7reg.r8;   arm7reg.r[9]=arm7reg.r9;
    	    arm7reg.r[10]=arm7reg.r10; arm7reg.r[11]=arm7reg.r11;
    	    arm7reg.r[12]=arm7reg.r12;
            arm7reg.r[13]=arm7reg.r13abt; arm7reg.r[14]=arm7reg.r14abt;
    	    break;
        case ARMMD_UND:
    	    arm7reg.r[8]=arm7reg.r8;   arm7reg.r[9]=arm7reg.r9;
    	    arm7reg.r[10]=arm7reg.r10; arm7reg.r[11]=arm7reg.r11;
    	    arm7reg.r[12]=arm7reg.r12;
            arm7reg.r[13]=arm7reg.r13und; arm7reg.r14und=arm7reg.r[14];
    	    break;
    	case ARMMD_FIQ:
            arm7reg.r[8]=arm7reg.r8fiq;   arm7reg.r[9]=arm7reg.r9fiq;
    	    arm7reg.r[10]=arm7reg.r10fiq; arm7reg.r[11]=arm7reg.r11fiq;
    	    arm7reg.r[12]=arm7reg.r12fiq; arm7reg.r[13]=arm7reg.r13fiq;
    	    arm7reg.r[14]=arm7reg.r14fiq;
    	    break;
    }
    arm7reg.cpsr|=ARMmodetocpsr[mdout];
    arm7reg.curmode=mdout;
}

void ARM7status(int offset, int mode)
{
    char str[512]; int a; u16 oph;
    u32 r15=arm7reg.r[15],r15orig=r15;
    static u32 dbgoldr[16],dbgoldf[5];
    RAMWORD op; u32 col;

    ARM7updateCPSR();
    dbgOutClear(dbuf,468,144);

    if((mode==3&&arm7reg.cpsr&ARMS_T)||(mode==2))
        r15-=(16-offset*2);
    else r15-=(32-offset*4);

    for(a=0;a<16;a++)
    {
        if(r15==r15orig) col=0x7E10; else col=0x7FFF;
        if((r15&0x0F000000)==(r15orig&0x0F000000)) op=MMUrdS(0,r15);
        switch(mode)
        {
            case 1:
                if((r15&0x0F000000)==(r15orig&0x0F000000))
                    sprintf(str,"%08X: %08X | %s",r15,op.data,ARM7DASM(op.data));
                else sprintf(str,"%08X",r15);
                r15+=4; break;
            case 2:
                if((r15&0x0F000000)==(r15orig&0x0F000000))
                {
                    if(r15&2) oph=op.data>>16; else oph=op.data&65535;
                    sprintf(str,"%08X: %04X | %s",r15,oph,Thumb7DASM(oph));
                }
                else sprintf(str,"%08X",r15);
                r15+=2; break;
            case 3:
                switch(arm7reg.cpsr&ARMS_T)
                {
                    case 0:
                        if((r15&0x0F000000)==(r15orig&0x0F000000))
                            sprintf(str,"%08X: %08X | %s",r15,op.data,ARM7DASM(op.data));
                        else sprintf(str,"%08X",r15);
                        r15+=4; break;
                    case ARMS_T:
                        if((r15&0x0F000000)==(r15orig&0x0F000000))
                        {
                            if(r15&2) oph=op.data>>16; else oph=op.data&65535;
                            sprintf(str,"%08X: %04X | %s",r15,oph,Thumb7DASM(oph));
                        }
                        else sprintf(str,"%08X",r15);
                        r15+=2; break;
                }
                break;
        }
        dbgOut(dbuf,468,144,str,0,16+a*8,col);
    }
    for(a=0;a<=15;a++)
    {
        sprintf(str,"r%02d: %08X",a,arm7reg.r[a]);
        if(arm7reg.r[a]!=dbgoldr[a])
        {
            dbgoldr[a]=arm7reg.r[a];
            dbgOut(dbuf,468,144,str,56*6,16+a*8,0x001F);
        }
	else dbgOut(dbuf,468,144,str,56*6,16+a*8,0x7FFF);
    }
    sprintf(str,"%s",ARMmodes[arm7reg.curmode]);
    dbgOut(dbuf, 468,144, str, 72*6,72,0x7FFF);

    a=(arm7reg.cpsr&ARMS_N)>>31; sprintf(str,"N: %d",a);
    if(a!=dbgoldf[0]) { dbgoldf[0]=a; dbgOut(dbuf,468,144,str,72*6,16,0x001F); }
    else dbgOut(dbuf,468,144,str,72*6,16,0x7FFF);

    a=(arm7reg.cpsr&ARMS_Z)>>30; sprintf(str,"Z: %d",a);
    if(a!=dbgoldf[1]) { dbgoldf[1]=a; dbgOut(dbuf,468,144,str,72*6,24,0x001F); }
    else dbgOut(dbuf,468,144,str,72*6,24,0x7FFF);

    a=(arm7reg.cpsr&ARMS_C)>>29; sprintf(str,"C: %d",a);
    if(a!=dbgoldf[2]) { dbgoldf[2]=a; dbgOut(dbuf,468,144,str,72*6,32,0x001F); }
    else dbgOut(dbuf,468,144,str,72*6,32,0x7FFF);

    a=(arm7reg.cpsr&ARMS_V)>>28; sprintf(str,"V: %d",a);
    if(a!=dbgoldf[3]) { dbgoldf[3]=a; dbgOut(dbuf,468,144,str,72*6,40,0x001F); }
    else dbgOut(dbuf,468,144,str,72*6,40,0x7FFF);

    a=(arm7reg.cpsr&ARMS_T)>>5;  sprintf(str,"T: %d",a);
    if(a!=dbgoldf[4]) { dbgoldf[4]=a; dbgOut(dbuf,468,144,str,72*6,56,0x001F); }
    else dbgOut(dbuf,468,144,str,72*6,56,0x7FFF);
}

//---Undefined/unimplemented opcode messaging------------------------------

void ARM7opUND()
{
    char str[80];
    sprintf(str,"Undefined opcode encountered: %08X.",arm7reg.curop);
    logvt->append(str);
}

void ARM7opUNP()
{
    char str[80];
    sprintf(str,"Unpredictable opcode encountered: %08X.",arm7reg.curop);
    logvt->append(str);
}

void ARM7opUNI()
{
    char str[80];
    sprintf(str,"Unimplemented opcode encountered: %08X.",arm7reg.curop);
    logvt->append(str);
}

void ARM7opUNL()
{
    RAMWORD op;
    u32 idx=((arm7reg.curop&0x0FF00000)>>16)|((arm7reg.curop&0x000000F0)>>4);
    u16 cond=(arm7reg.curop&0xF0000000)>>28;
    op.data=arm7reg.curop;
    op.op=arm7ops[idx].op;
    op.tidxlo=(arm7reg.curop>>8)&255;
    op.tidxhi=(arm7reg.curop>>24);
    op.cond=cond;
    op.cyc=arm7ops[idx].cyc;
    op.flags=0;
    MMUwrS(0,arm7reg.r[15]-4,op);

	if(arm7reg.cpsr&ARMS_T)
	{
            switch(arm7reg.r[15]&2)
            {
                case 0:
		    arm7reg.curop=op.data>>16;
		    thumb7ops[arm7reg.curop>>8].op();
                    arm7reg.clock+=thumb7ops[arm7reg.curop>>8].cyc; break;
                case 2:
		    arm7reg.curop=op.data&65535;
		    thumb7ops[arm7reg.curop>>8].op();
                    arm7reg.clock+=thumb7ops[arm7reg.curop>>8].cyc; break;
            }
        } else {
            if(ARM7cond[op.cond]())
            {
                arm7reg.curop=op.data;
                op.op();
                arm7reg.clock+=op.cyc;
            }
        }
}

//---Conditional execution implementation----------------------------------

int ARM7condEQ(){return  arm7reg.flags[ARMFLAG_Z];}
int ARM7condNE(){return !arm7reg.flags[ARMFLAG_Z];}
int ARM7condCS(){return  arm7reg.flags[ARMFLAG_C];}
int ARM7condCC(){return !arm7reg.flags[ARMFLAG_C];}
int ARM7condMI(){return  arm7reg.flags[ARMFLAG_N];}
int ARM7condPL(){return !arm7reg.flags[ARMFLAG_N];}
int ARM7condVS(){return  arm7reg.flags[ARMFLAG_V];}
int ARM7condVC(){return !arm7reg.flags[ARMFLAG_V];}

int ARM7condHI(){return (!arm7reg.flags[ARMFLAG_Z])&arm7reg.flags[ARMFLAG_C];}
int ARM7condLS(){return arm7reg.flags[ARMFLAG_Z]|(!arm7reg.flags[ARMFLAG_C]);}
int ARM7condGE(){return !(arm7reg.flags[ARMFLAG_N]^arm7reg.flags[ARMFLAG_V]);}
int ARM7condLT(){return  (arm7reg.flags[ARMFLAG_N]^arm7reg.flags[ARMFLAG_V]);}
int ARM7condGT(){return (!arm7reg.flags[ARMFLAG_Z])&(!(arm7reg.flags[ARMFLAG_N]^arm7reg.flags[ARMFLAG_V]));}
int ARM7condLE(){return arm7reg.flags[ARMFLAG_Z]|(arm7reg.flags[ARMFLAG_N]^arm7reg.flags[ARMFLAG_V]);}
int ARM7condAL(){return 1;}
int ARM7condNV(){return 0;}

//---Branching-------------------------------------------------------------

#define ARM7addrB() \
    arm7reg.tmp2=arm7reg.curop&0x00FFFFFF; \
    arm7reg.tmp1=(arm7reg.tmp2&0x00800000)? \
                 (0xFF000000|arm7reg.tmp2): \
		 (0x00000000|arm7reg.tmp2)

void ARM7opB()
{
    ARM7addrB();
    arm7reg.r[15]+=((signed)arm7reg.tmp1)*4+4;
}

void ARM7opBL()
{
    ARM7addrB();
    arm7reg.r[14]=arm7reg.r[15];
    arm7reg.r[15]+=((signed)arm7reg.tmp1)*4+4;
}

void ARM7opBX()
{
    ARM7addrB();
    arm7reg.r[15]=arm7reg.r[ARM7OP_RM]&0xFFFFFFFE;
    if(arm7reg.r[ARM7OP_RM]&1) arm7reg.cpsr|=ARMS_T;
                          else arm7reg.cpsr&=(0xFFFFFFFF-ARMS_T);
}

void ARM7opBLX()
{
  // Implemented in ARM5
}

void ARM7opBLXr()
{
    ARM7addrB();
    arm7reg.r[14]=arm7reg.r[15];
    arm7reg.r[15]=arm7reg.r[ARM7OP_RM]&0xFFFFFFFE;
    if(arm7reg.r[ARM7OP_RM]&1) arm7reg.cpsr|=ARMS_T;
                          else arm7reg.cpsr&=(0xFFFFFFFF-ARMS_T);
}

//---Data Processing addressing modes, opcodes-----------------------------

#define ARM7addrDPreg() \
    arm7reg.tmp1=arm7reg.r[ARM7OP_RM]; if(ARM7OP_RM==15) arm7reg.tmp1+=4; \
    arm7reg.tmp2=arm7reg.r[ARM7OP_RN]; if(ARM7OP_RN==15) arm7reg.tmp2+=4; \
    arm7reg.tmpc=arm7reg.flags[ARMFLAG_C]

#define ARM7addrDPimm() \
    arm7reg.tmp3 = arm7reg.curop&255; \
    arm7reg.tmp4 = (arm7reg.curop&0x00000F00)>>7; \
    arm7reg.tmp1 = (arm7reg.tmp3>>arm7reg.tmp4)|((arm7reg.tmp3&((1<<arm7reg.tmp4)-1))<<(32-arm7reg.tmp4)); \
    arm7reg.tmp2 = arm7reg.r[ARM7OP_RN]; if(ARM7OP_RN==15) arm7reg.tmp2+=4; \
    if(arm7reg.tmp4) arm7reg.tmpc = (arm7reg.tmp1&0x80000000)>>31; \
    else arm7reg.tmpc = arm7reg.flags[ARMFLAG_C]

#define ARM7addrDPlli() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp2 = arm7reg.r[ARM7OP_RN]; if(ARM7OP_RN==15) arm7reg.tmp2+=4; \
    arm7reg.tmp1 = arm7reg.r[ARM7OP_RM]; if(ARM7OP_RM==15) arm7reg.tmp1+=4; \
    if(!arm7reg.tmp4) arm7reg.tmpc = arm7reg.flags[ARMFLAG_C]; \
    else { \
        arm7reg.tmpc = (arm7reg.tmp1 >> (32-arm7reg.tmp4))&1; \
        arm7reg.tmp1 <<= arm7reg.tmp4; \
    }

#define ARM7addrDPllr() \
    arm7reg.tmp4=arm7reg.r[ARM7OP_RS]&255; \
    arm7reg.tmp2=arm7reg.r[ARM7OP_RN]; if(ARM7OP_RN==15) arm7reg.tmp2+=4; \
    arm7reg.tmp1=arm7reg.r[ARM7OP_RM]; if(ARM7OP_RM==15) arm7reg.tmp1+=4; \
    if(!arm7reg.tmp4) arm7reg.tmpc = arm7reg.flags[ARMFLAG_C]; \
    else if(arm7reg.tmp4<=32) { \
        arm7reg.tmpc = (arm7reg.tmp1 >> (32-arm7reg.tmp4))&1; \
        arm7reg.tmp1 <<= arm7reg.tmp4; \
    } else arm7reg.tmp1=0; arm7reg.tmpc=0;

#define ARM7addrDPlri() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp2=arm7reg.r[ARM7OP_RN]; if(ARM7OP_RN==15) arm7reg.tmp2+=4; \
    arm7reg.tmp1=arm7reg.r[ARM7OP_RM]; if(ARM7OP_RM==15) arm7reg.tmp1+=4; \
    if(!arm7reg.tmp4) { \
        arm7reg.tmpc=arm7reg.tmp1 >> 31; arm7reg.tmp1=0; \
    } else { \
        arm7reg.tmpc = (arm7reg.tmp1 >> (arm7reg.tmp4-1))&1; \
        arm7reg.tmp1 >>= arm7reg.tmp4; \
    }

#define ARM7addrDPlrr() \
    arm7reg.tmp4=arm7reg.r[ARM7OP_RS]&255; \
    arm7reg.tmp2=arm7reg.r[ARM7OP_RN]; if(ARM7OP_RN==15) arm7reg.tmp2+=4; \
    arm7reg.tmp1=arm7reg.r[ARM7OP_RM]; if(ARM7OP_RM==15) arm7reg.tmp1+=4; \
    if(!arm7reg.tmp4) arm7reg.tmpc = arm7reg.flags[ARMFLAG_C]; \
    else if(arm7reg.tmp4<=32) { \
        arm7reg.tmpc = (arm7reg.tmp1 >> (arm7reg.tmp4-1))&1; \
        arm7reg.tmp1 >>= arm7reg.tmp4; \
    } else { \
        arm7reg.tmp1 = 0; arm7reg.tmpc = 0; \
    }

#define ARM7addrDPari() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp2=arm7reg.r[ARM7OP_RN]; if(ARM7OP_RN==15) arm7reg.tmp2+=4; \
    arm7reg.tmp1=arm7reg.r[ARM7OP_RM]; if(ARM7OP_RM==15) arm7reg.tmp1+=4; \
    if(!arm7reg.tmp4) { \
        arm7reg.tmpc = arm7reg.tmp1 >> 31; \
        if(arm7reg.tmp1&0x80000000) arm7reg.tmp1 = 0xFFFFFFFF; \
	else arm7reg.tmp1=0; \
    } else { \
        arm7reg.tmpc = (arm7reg.tmp1 >> (arm7reg.tmp4-1))&1; \
        arm7reg.tmp1 = (signed)arm7reg.tmp1 >> arm7reg.tmp4; \
    }

#define ARM7addrDParr() \
    arm7reg.tmp4=arm7reg.r[ARM7OP_RS]&255; \
    arm7reg.tmp2=arm7reg.r[ARM7OP_RN]; if(ARM7OP_RN==15) arm7reg.tmp2+=4; \
    arm7reg.tmp1=arm7reg.r[ARM7OP_RM]; if(ARM7OP_RM==15) arm7reg.tmp1+=4; \
    if(!arm7reg.tmp4) arm7reg.tmpc = arm7reg.flags[ARMFLAG_C]; \
    else if(arm7reg.tmp4<32) { \
        arm7reg.tmpc = (arm7reg.tmp1 >> (arm7reg.tmp4-1))&1; \
        arm7reg.tmp1 = (signed)arm7reg.tmp1 >> arm7reg.tmp4; \
    } else { \
        arm7reg.tmpc = arm7reg.tmp1 >> 31; \
        if(arm7reg.tmp1&0x80000000) arm7reg.tmp1 = 0xFFFFFFFF; \
	else arm7reg.tmp1=0; \
    }

#define ARM7addrDPrri() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp2=arm7reg.r[ARM7OP_RN]; if(ARM7OP_RN==15) arm7reg.tmp2+=4; \
    arm7reg.tmp3=arm7reg.r[ARM7OP_RM]; if(ARM7OP_RM==15) arm7reg.tmp1+=4; \
    if(!arm7reg.tmp4) { \
        arm7reg.tmp1 = (arm7reg.flags[ARMFLAG_C]<<31)|(arm7reg.tmp3>>1); \
        arm7reg.tmpc = arm7reg.tmp3&1; \
    } else { \
        arm7reg.tmp1 = (arm7reg.tmp3>>arm7reg.tmp4)|((arm7reg.tmp3&((1<<arm7reg.tmp4)-1))<<(32-arm7reg.tmp4)); \
        arm7reg.tmpc = (arm7reg.tmp3 >> (arm7reg.tmp4-1))&1; \
    }

#define ARM7addrDPrrr() \
    arm7reg.tmp4=arm7reg.r[ARM7OP_RS]&255; \
    arm7reg.tmp2=arm7reg.r[ARM7OP_RN]; if(ARM7OP_RN==15) arm7reg.tmp2+=4; \
    arm7reg.tmp3=arm7reg.r[ARM7OP_RM]; if(ARM7OP_RM==15) arm7reg.tmp1+=4; \
    arm7reg.tmp1=arm7reg.tmp3; \
    if(!arm7reg.tmp4) arm7reg.tmpc = arm7reg.flags[ARMFLAG_C]; \
    else if(!(arm7reg.tmp4&31)) arm7reg.tmpc = arm7reg.tmp3 >> 31; \
    else { \
        arm7reg.tmp4&=31; \
        arm7reg.tmp1 = (arm7reg.tmp3>>arm7reg.tmp4)|((arm7reg.tmp3&((1<<arm7reg.tmp4)-1))<<(32-arm7reg.tmp4)); \
        arm7reg.tmpc = (arm7reg.tmp3 >> (arm7reg.tmp4-1))&1; \
    }

#define ARM7opAND() arm7reg.r[ARM7OP_RD]=arm7reg.tmp1 & arm7reg.tmp2
#define ARM7opEOR() arm7reg.r[ARM7OP_RD]=arm7reg.tmp1 ^ arm7reg.tmp2
#define ARM7opSUB() arm7reg.r[ARM7OP_RD]=arm7reg.tmp2 - arm7reg.tmp1
#define ARM7opRSB() arm7reg.r[ARM7OP_RD]=arm7reg.tmp1 - arm7reg.tmp2
#define ARM7opADD() arm7reg.r[ARM7OP_RD]=arm7reg.tmp1 + arm7reg.tmp2
#define ARM7opADC() arm7reg.r[ARM7OP_RD]=arm7reg.tmp1 + arm7reg.tmp2 + arm7reg.tmpc
#define ARM7opSBC() arm7reg.r[ARM7OP_RD]=arm7reg.tmp2 - arm7reg.tmp1 - arm7reg.tmpc
#define ARM7opRSC() arm7reg.r[ARM7OP_RD]=arm7reg.tmp1 - arm7reg.tmp2 + arm7reg.tmpc
#define ARM7opTST() arm7reg.tmp3=arm7reg.tmp1 & arm7reg.tmp2
#define ARM7opTEQ() arm7reg.tmp3=arm7reg.tmp1 ^ arm7reg.tmp2
#define ARM7opCMP() arm7reg.tmp3=arm7reg.tmp2 - arm7reg.tmp1
#define ARM7opCMN() arm7reg.tmp3=arm7reg.tmp1 + arm7reg.tmp2
#define ARM7opORR() arm7reg.r[ARM7OP_RD]=arm7reg.tmp1 | arm7reg.tmp2
#define ARM7opMOV() arm7reg.r[ARM7OP_RD]=arm7reg.tmp1
#define ARM7opBIC() arm7reg.r[ARM7OP_RD]=arm7reg.tmp2 & ~(arm7reg.tmp1)
#define ARM7opMVN() arm7reg.r[ARM7OP_RD]=~(arm7reg.tmp1)

#define ARM7addrDPflagBIT() \
    if(ARM7OP_RD==15 && arm7reg.curmode>1) \
    { \
        arm7reg.tmp1=arm7reg.spsr[arm7reg.curmode]; \
	if((arm7reg.cpsr&15)!=(arm7reg.tmp1&15)) \
	{ \
            ARM7modesw(ARMcpsrtomode[arm7reg.cpsr&15], \
                       ARMcpsrtomode[arm7reg.tmp1&15]); \
            arm7reg.cpsr&=0x0000001F; \
            arm7reg.cpsr|=(arm7reg.tmp1&0xFFFFFFE0); \
	} \
	else arm7reg.cpsr=arm7reg.tmp1; \
        ARM7splitCPSR(); \
    } \
    else { \
        arm7reg.flags[ARMFLAG_N]=arm7reg.r[ARM7OP_RD]>>31; \
        arm7reg.flags[ARMFLAG_Z]=arm7reg.r[ARM7OP_RD]?0:1; \
        arm7reg.flags[ARMFLAG_C]=arm7reg.tmpc; \
    }

#define ARM7addrDPflagADD() \
    if(ARM7OP_RD==15 && arm7reg.curmode>1) \
    { \
        arm7reg.tmp1=arm7reg.spsr[arm7reg.curmode]; \
	if((arm7reg.cpsr&15)!=(arm7reg.tmp1&15)) \
	{ \
            ARM7modesw(ARMcpsrtomode[arm7reg.cpsr&15], \
                       ARMcpsrtomode[arm7reg.tmp1&15]); \
            arm7reg.cpsr&=0x0000001F; \
            arm7reg.cpsr|=(arm7reg.tmp1&0xFFFFFFE0); \
	} \
	else arm7reg.cpsr=arm7reg.tmp1; \
        ARM7splitCPSR(); \
    } \
    else { \
        arm7reg.flags[ARMFLAG_N]=arm7reg.r[ARM7OP_RD]>>31; \
        arm7reg.flags[ARMFLAG_Z]=arm7reg.r[ARM7OP_RD]?0:1; \
        arm7reg.flags[ARMFLAG_V]=((arm7reg.tmp2^arm7reg.r[ARM7OP_RD])&(arm7reg.tmp1^arm7reg.r[ARM7OP_RD]))>>31; \
        arm7reg.flags[ARMFLAG_C]=((arm7reg.tmp2>>31)+(arm7reg.tmp1>>31)+arm7reg.flags[ARMFLAG_V])>>1; \
    }

#define ARM7addrDPflagADC() \
    if(ARM7OP_RD==15 && arm7reg.curmode>1) \
    { \
        arm7reg.tmp1=arm7reg.spsr[arm7reg.curmode]; \
	if((arm7reg.cpsr&15)!=(arm7reg.tmp1&15)) \
	{ \
            ARM7modesw(ARMcpsrtomode[arm7reg.cpsr&15], \
                       ARMcpsrtomode[arm7reg.tmp1&15]); \
            arm7reg.cpsr&=0x0000001F; \
            arm7reg.cpsr|=(arm7reg.tmp1&0xFFFFFFE0); \
	} \
	else arm7reg.cpsr=arm7reg.tmp1; \
        ARM7splitCPSR(); \
    } \
    else { \
        arm7reg.flags[ARMFLAG_N]=arm7reg.r[ARM7OP_RD]>>31; \
        arm7reg.flags[ARMFLAG_Z]=arm7reg.r[ARM7OP_RD]?0:1; \
        arm7reg.tmp2+=arm7reg.tmpc; \
        arm7reg.flags[ARMFLAG_V]=((arm7reg.tmp2^arm7reg.r[ARM7OP_RD])&(arm7reg.tmp1^arm7reg.r[ARM7OP_RD]))>>31; \
        arm7reg.flags[ARMFLAG_C]=((arm7reg.tmp2>>31)+(arm7reg.tmp1>>31)+arm7reg.flags[ARMFLAG_V])>>1; \
    }

#define ARM7addrDPflagSUB() \
    if(ARM7OP_RD==15 && arm7reg.curmode>1) \
    { \
        arm7reg.tmp1=arm7reg.spsr[arm7reg.curmode]; \
	if((arm7reg.cpsr&15)!=(arm7reg.tmp1&15)) \
	{ \
            ARM7modesw(ARMcpsrtomode[arm7reg.cpsr&15], \
                       ARMcpsrtomode[arm7reg.tmp1&15]); \
            arm7reg.cpsr&=0x0000001F; \
            arm7reg.cpsr|=(arm7reg.tmp1&0xFFFFFFE0); \
	} \
	else arm7reg.cpsr=arm7reg.tmp1; \
        ARM7splitCPSR(); \
    } \
    else { \
        arm7reg.flags[ARMFLAG_N]=arm7reg.r[ARM7OP_RD]>>31; \
        arm7reg.flags[ARMFLAG_Z]=arm7reg.r[ARM7OP_RD]?0:1; \
        arm7reg.flags[ARMFLAG_C]=(arm7reg.tmp2<arm7reg.tmp1)?0:1; \
        arm7reg.flags[ARMFLAG_V]=((arm7reg.tmp2^arm7reg.tmp1)&(arm7reg.tmp2^arm7reg.r[ARM7OP_RD]))>>31; \
    }

#define ARM7addrDPflagSBC() \
    if(ARM7OP_RD==15 && arm7reg.curmode>1) \
    { \
        arm7reg.tmp1=arm7reg.spsr[arm7reg.curmode]; \
	if((arm7reg.cpsr&15)!=(arm7reg.tmp1&15)) \
	{ \
            ARM7modesw(ARMcpsrtomode[arm7reg.cpsr&15], \
                       ARMcpsrtomode[arm7reg.tmp1&15]); \
            arm7reg.cpsr&=0x0000001F; \
            arm7reg.cpsr|=(arm7reg.tmp1&0xFFFFFFE0); \
	} \
	else arm7reg.cpsr=arm7reg.tmp1; \
        ARM7splitCPSR(); \
    } \
    else { \
        arm7reg.flags[ARMFLAG_N]=arm7reg.r[ARM7OP_RD]>>31; \
        arm7reg.flags[ARMFLAG_Z]=arm7reg.r[ARM7OP_RD]?0:1; \
        arm7reg.flags[ARMFLAG_C]=(arm7reg.tmp2<(arm7reg.tmp1-arm7reg.tmpc))?0:1; \
        arm7reg.tmp1+=arm7reg.tmpc;\
        arm7reg.flags[ARMFLAG_V]=((arm7reg.tmp2^arm7reg.tmp1)&(arm7reg.tmp2^arm7reg.r[ARM7OP_RD]))>>31; \
    }

#define ARM7addrDPflagRSB() \
    if(ARM7OP_RD==15 && arm7reg.curmode>1) \
    { \
        arm7reg.tmp1=arm7reg.spsr[arm7reg.curmode]; \
	if((arm7reg.cpsr&15)!=(arm7reg.tmp1&15)) \
	{ \
            ARM7modesw(ARMcpsrtomode[arm7reg.cpsr&15], \
                       ARMcpsrtomode[arm7reg.tmp1&15]); \
            arm7reg.cpsr&=0x0000001F; \
            arm7reg.cpsr|=(arm7reg.tmp1&0xFFFFFFE0); \
	} \
	else arm7reg.cpsr=arm7reg.tmp1; \
        ARM7splitCPSR(); \
    } \
    else { \
        arm7reg.flags[ARMFLAG_N]=arm7reg.r[ARM7OP_RD]>>31; \
        arm7reg.flags[ARMFLAG_Z]=arm7reg.r[ARM7OP_RD]?0:1; \
        arm7reg.flags[ARMFLAG_C]=(arm7reg.tmp1<arm7reg.tmp2)?0:1; \
        arm7reg.flags[ARMFLAG_V]=((arm7reg.tmp2^arm7reg.tmp1)&(arm7reg.tmp2^arm7reg.r[ARM7OP_RD]))>>31; \
    }

#define ARM7addrDPflagRSC() \
    if(ARM7OP_RD==15 && arm7reg.curmode>1) \
    { \
        arm7reg.tmp1=arm7reg.spsr[arm7reg.curmode]; \
	if((arm7reg.cpsr&15)!=(arm7reg.tmp1&15)) \
	{ \
            ARM7modesw(ARMcpsrtomode[arm7reg.cpsr&15], \
                       ARMcpsrtomode[arm7reg.tmp1&15]); \
            arm7reg.cpsr&=0x0000001F; \
            arm7reg.cpsr|=(arm7reg.tmp1&0xFFFFFFE0); \
	} \
	else arm7reg.cpsr=arm7reg.tmp1; \
        ARM7splitCPSR(); \
    } \
    else { \
        arm7reg.flags[ARMFLAG_N]=arm7reg.r[ARM7OP_RD]>>31; \
        arm7reg.flags[ARMFLAG_Z]=arm7reg.r[ARM7OP_RD]?0:1; \
        arm7reg.flags[ARMFLAG_C]=(arm7reg.tmp1<(arm7reg.tmp2-(~arm7reg.tmpc)))?0:1; \
        arm7reg.tmp2-=arm7reg.tmpc;\
        arm7reg.flags[ARMFLAG_V]=((arm7reg.tmp2^arm7reg.tmp1)&(arm7reg.tmp2^arm7reg.r[ARM7OP_RD]))>>31; \
    }

#define ARM7addrDPflagCMN() \
    arm7reg.flags[ARMFLAG_N]=arm7reg.tmp3>>31; \
    arm7reg.flags[ARMFLAG_Z]=arm7reg.tmp3?0:1; \
    arm7reg.flags[ARMFLAG_V]=((arm7reg.tmp2^arm7reg.tmp3)&(arm7reg.tmp1^arm7reg.tmp3))>>31; \
    arm7reg.flags[ARMFLAG_C]=((arm7reg.tmp2>>31)+(arm7reg.tmp1>>31)+arm7reg.flags[ARMFLAG_V])>>1

#define ARM7addrDPflagCMP() \
    arm7reg.flags[ARMFLAG_N]=arm7reg.tmp3>>31; \
    arm7reg.flags[ARMFLAG_Z]=arm7reg.tmp3?0:1; \
    arm7reg.flags[ARMFLAG_C]=(arm7reg.tmp2<arm7reg.tmp1)?0:1; \
    arm7reg.flags[ARMFLAG_V]=((arm7reg.tmp2^arm7reg.tmp1)&(arm7reg.tmp2^arm7reg.tmp3))>>31

#define ARM7addrDPflagTST() \
    arm7reg.flags[ARMFLAG_N]=arm7reg.tmp3>>31; \
    arm7reg.flags[ARMFLAG_Z]=arm7reg.tmp3?0:1; \
    arm7reg.flags[ARMFLAG_C]=arm7reg.tmpc

void ARM7opANDreg() { ARM7addrDPreg(); ARM7opAND(); }
void ARM7opANDimm() { ARM7addrDPimm(); ARM7opAND(); }
void ARM7opANDlli() { ARM7addrDPlli(); ARM7opAND(); }
void ARM7opANDllr() { ARM7addrDPllr(); ARM7opAND(); }
void ARM7opANDlri() { ARM7addrDPlri(); ARM7opAND(); }
void ARM7opANDlrr() { ARM7addrDPlrr(); ARM7opAND(); }
void ARM7opANDari() { ARM7addrDPari(); ARM7opAND(); }
void ARM7opANDarr() { ARM7addrDParr(); ARM7opAND(); }
void ARM7opANDrri() { ARM7addrDPrri(); ARM7opAND(); }
void ARM7opANDrrr() { ARM7addrDPrrr(); ARM7opAND(); }

void ARM7opEORreg() { ARM7addrDPreg(); ARM7opEOR(); }
void ARM7opEORimm() { ARM7addrDPimm(); ARM7opEOR(); }
void ARM7opEORlli() { ARM7addrDPlli(); ARM7opEOR(); }
void ARM7opEORllr() { ARM7addrDPllr(); ARM7opEOR(); }
void ARM7opEORlri() { ARM7addrDPlri(); ARM7opEOR(); }
void ARM7opEORlrr() { ARM7addrDPlrr(); ARM7opEOR(); }
void ARM7opEORari() { ARM7addrDPari(); ARM7opEOR(); }
void ARM7opEORarr() { ARM7addrDParr(); ARM7opEOR(); }
void ARM7opEORrri() { ARM7addrDPrri(); ARM7opEOR(); }
void ARM7opEORrrr() { ARM7addrDPrrr(); ARM7opEOR(); }

void ARM7opSUBreg() { ARM7addrDPreg(); ARM7opSUB(); }
void ARM7opSUBimm() { ARM7addrDPimm(); ARM7opSUB(); }
void ARM7opSUBlli() { ARM7addrDPlli(); ARM7opSUB(); }
void ARM7opSUBllr() { ARM7addrDPllr(); ARM7opSUB(); }
void ARM7opSUBlri() { ARM7addrDPlri(); ARM7opSUB(); }
void ARM7opSUBlrr() { ARM7addrDPlrr(); ARM7opSUB(); }
void ARM7opSUBari() { ARM7addrDPari(); ARM7opSUB(); }
void ARM7opSUBarr() { ARM7addrDParr(); ARM7opSUB(); }
void ARM7opSUBrri() { ARM7addrDPrri(); ARM7opSUB(); }
void ARM7opSUBrrr() { ARM7addrDPrrr(); ARM7opSUB(); }

void ARM7opRSBreg() { ARM7addrDPreg(); ARM7opRSB(); }
void ARM7opRSBimm() { ARM7addrDPimm(); ARM7opRSB(); }
void ARM7opRSBlli() { ARM7addrDPlli(); ARM7opRSB(); }
void ARM7opRSBllr() { ARM7addrDPllr(); ARM7opRSB(); }
void ARM7opRSBlri() { ARM7addrDPlri(); ARM7opRSB(); }
void ARM7opRSBlrr() { ARM7addrDPlrr(); ARM7opRSB(); }
void ARM7opRSBari() { ARM7addrDPari(); ARM7opRSB(); }
void ARM7opRSBarr() { ARM7addrDParr(); ARM7opRSB(); }
void ARM7opRSBrri() { ARM7addrDPrri(); ARM7opRSB(); }
void ARM7opRSBrrr() { ARM7addrDPrrr(); ARM7opRSB(); }

void ARM7opADDreg() { ARM7addrDPreg(); ARM7opADD(); }
void ARM7opADDimm() { ARM7addrDPimm(); ARM7opADD(); }
void ARM7opADDlli() { ARM7addrDPlli(); ARM7opADD(); }
void ARM7opADDllr() { ARM7addrDPllr(); ARM7opADD(); }
void ARM7opADDlri() { ARM7addrDPlri(); ARM7opADD(); }
void ARM7opADDlrr() { ARM7addrDPlrr(); ARM7opADD(); }
void ARM7opADDari() { ARM7addrDPari(); ARM7opADD(); }
void ARM7opADDarr() { ARM7addrDParr(); ARM7opADD(); }
void ARM7opADDrri() { ARM7addrDPrri(); ARM7opADD(); }
void ARM7opADDrrr() { ARM7addrDPrrr(); ARM7opADD(); }

void ARM7opADCreg() { ARM7addrDPreg(); ARM7opADC(); }
void ARM7opADCimm() { ARM7addrDPimm(); ARM7opADC(); }
void ARM7opADClli() { ARM7addrDPlli(); ARM7opADC(); }
void ARM7opADCllr() { ARM7addrDPllr(); ARM7opADC(); }
void ARM7opADClri() { ARM7addrDPlri(); ARM7opADC(); }
void ARM7opADClrr() { ARM7addrDPlrr(); ARM7opADC(); }
void ARM7opADCari() { ARM7addrDPari(); ARM7opADC(); }
void ARM7opADCarr() { ARM7addrDParr(); ARM7opADC(); }
void ARM7opADCrri() { ARM7addrDPrri(); ARM7opADC(); }
void ARM7opADCrrr() { ARM7addrDPrrr(); ARM7opADC(); }

void ARM7opSBCreg() { ARM7addrDPreg(); ARM7opSBC(); }
void ARM7opSBCimm() { ARM7addrDPimm(); ARM7opSBC(); }
void ARM7opSBClli() { ARM7addrDPlli(); ARM7opSBC(); }
void ARM7opSBCllr() { ARM7addrDPllr(); ARM7opSBC(); }
void ARM7opSBClri() { ARM7addrDPlri(); ARM7opSBC(); }
void ARM7opSBClrr() { ARM7addrDPlrr(); ARM7opSBC(); }
void ARM7opSBCari() { ARM7addrDPari(); ARM7opSBC(); }
void ARM7opSBCarr() { ARM7addrDParr(); ARM7opSBC(); }
void ARM7opSBCrri() { ARM7addrDPrri(); ARM7opSBC(); }
void ARM7opSBCrrr() { ARM7addrDPrrr(); ARM7opSBC(); }

void ARM7opRSCreg() { ARM7addrDPreg(); ARM7opRSC(); }
void ARM7opRSCimm() { ARM7addrDPimm(); ARM7opRSC(); }
void ARM7opRSClli() { ARM7addrDPlli(); ARM7opRSC(); }
void ARM7opRSCllr() { ARM7addrDPllr(); ARM7opRSC(); }
void ARM7opRSClri() { ARM7addrDPlri(); ARM7opRSC(); }
void ARM7opRSClrr() { ARM7addrDPlrr(); ARM7opRSC(); }
void ARM7opRSCari() { ARM7addrDPari(); ARM7opRSC(); }
void ARM7opRSCarr() { ARM7addrDParr(); ARM7opRSC(); }
void ARM7opRSCrri() { ARM7addrDPrri(); ARM7opRSC(); }
void ARM7opRSCrrr() { ARM7addrDPrrr(); ARM7opRSC(); }

void ARM7opTSTreg() { ARM7addrDPreg(); ARM7opTST(); }
void ARM7opTSTimm() { ARM7addrDPimm(); ARM7opTST(); }
void ARM7opTSTlli() { ARM7addrDPlli(); ARM7opTST(); }
void ARM7opTSTllr() { ARM7addrDPllr(); ARM7opTST(); }
void ARM7opTSTlri() { ARM7addrDPlri(); ARM7opTST(); }
void ARM7opTSTlrr() { ARM7addrDPlrr(); ARM7opTST(); }
void ARM7opTSTari() { ARM7addrDPari(); ARM7opTST(); }
void ARM7opTSTarr() { ARM7addrDParr(); ARM7opTST(); }
void ARM7opTSTrri() { ARM7addrDPrri(); ARM7opTST(); }
void ARM7opTSTrrr() { ARM7addrDPrrr(); ARM7opTST(); }

void ARM7opTEQreg() { ARM7addrDPreg(); ARM7opTEQ(); }
void ARM7opTEQimm() { ARM7addrDPimm(); ARM7opTEQ(); }
void ARM7opTEQlli() { ARM7addrDPlli(); ARM7opTEQ(); }
void ARM7opTEQllr() { ARM7addrDPllr(); ARM7opTEQ(); }
void ARM7opTEQlri() { ARM7addrDPlri(); ARM7opTEQ(); }
void ARM7opTEQlrr() { ARM7addrDPlrr(); ARM7opTEQ(); }
void ARM7opTEQari() { ARM7addrDPari(); ARM7opTEQ(); }
void ARM7opTEQarr() { ARM7addrDParr(); ARM7opTEQ(); }
void ARM7opTEQrri() { ARM7addrDPrri(); ARM7opTEQ(); }
void ARM7opTEQrrr() { ARM7addrDPrrr(); ARM7opTEQ(); }

void ARM7opCMPreg() { ARM7addrDPreg(); ARM7opCMP(); }
void ARM7opCMPimm() { ARM7addrDPimm(); ARM7opCMP(); }
void ARM7opCMPlli() { ARM7addrDPlli(); ARM7opCMP(); }
void ARM7opCMPllr() { ARM7addrDPllr(); ARM7opCMP(); }
void ARM7opCMPlri() { ARM7addrDPlri(); ARM7opCMP(); }
void ARM7opCMPlrr() { ARM7addrDPlrr(); ARM7opCMP(); }
void ARM7opCMPari() { ARM7addrDPari(); ARM7opCMP(); }
void ARM7opCMParr() { ARM7addrDParr(); ARM7opCMP(); }
void ARM7opCMPrri() { ARM7addrDPrri(); ARM7opCMP(); }
void ARM7opCMPrrr() { ARM7addrDPrrr(); ARM7opCMP(); }

void ARM7opCMNreg() { ARM7addrDPreg(); ARM7opCMN(); }
void ARM7opCMNimm() { ARM7addrDPimm(); ARM7opCMN(); }
void ARM7opCMNlli() { ARM7addrDPlli(); ARM7opCMN(); }
void ARM7opCMNllr() { ARM7addrDPllr(); ARM7opCMN(); }
void ARM7opCMNlri() { ARM7addrDPlri(); ARM7opCMN(); }
void ARM7opCMNlrr() { ARM7addrDPlrr(); ARM7opCMN(); }
void ARM7opCMNari() { ARM7addrDPari(); ARM7opCMN(); }
void ARM7opCMNarr() { ARM7addrDParr(); ARM7opCMN(); }
void ARM7opCMNrri() { ARM7addrDPrri(); ARM7opCMN(); }
void ARM7opCMNrrr() { ARM7addrDPrrr(); ARM7opCMN(); }

void ARM7opORRreg() { ARM7addrDPreg(); ARM7opORR(); }
void ARM7opORRimm() { ARM7addrDPimm(); ARM7opORR(); }
void ARM7opORRlli() { ARM7addrDPlli(); ARM7opORR(); }
void ARM7opORRllr() { ARM7addrDPllr(); ARM7opORR(); }
void ARM7opORRlri() { ARM7addrDPlri(); ARM7opORR(); }
void ARM7opORRlrr() { ARM7addrDPlrr(); ARM7opORR(); }
void ARM7opORRari() { ARM7addrDPari(); ARM7opORR(); }
void ARM7opORRarr() { ARM7addrDParr(); ARM7opORR(); }
void ARM7opORRrri() { ARM7addrDPrri(); ARM7opORR(); }
void ARM7opORRrrr() { ARM7addrDPrrr(); ARM7opORR(); }

void ARM7opMOVreg() { ARM7addrDPreg(); ARM7opMOV(); }
void ARM7opMOVimm() { ARM7addrDPimm(); ARM7opMOV(); }
void ARM7opMOVlli() { ARM7addrDPlli(); ARM7opMOV(); }
void ARM7opMOVllr() { ARM7addrDPllr(); ARM7opMOV(); }
void ARM7opMOVlri() { ARM7addrDPlri(); ARM7opMOV(); }
void ARM7opMOVlrr() { ARM7addrDPlrr(); ARM7opMOV(); }
void ARM7opMOVari() { ARM7addrDPari(); ARM7opMOV(); }
void ARM7opMOVarr() { ARM7addrDParr(); ARM7opMOV(); }
void ARM7opMOVrri() { ARM7addrDPrri(); ARM7opMOV(); }
void ARM7opMOVrrr() { ARM7addrDPrrr(); ARM7opMOV(); }

void ARM7opBICreg() { ARM7addrDPreg(); ARM7opBIC(); }
void ARM7opBICimm() { ARM7addrDPimm(); ARM7opBIC(); }
void ARM7opBIClli() { ARM7addrDPlli(); ARM7opBIC(); }
void ARM7opBICllr() { ARM7addrDPllr(); ARM7opBIC(); }
void ARM7opBIClri() { ARM7addrDPlri(); ARM7opBIC(); }
void ARM7opBIClrr() { ARM7addrDPlrr(); ARM7opBIC(); }
void ARM7opBICari() { ARM7addrDPari(); ARM7opBIC(); }
void ARM7opBICarr() { ARM7addrDParr(); ARM7opBIC(); }
void ARM7opBICrri() { ARM7addrDPrri(); ARM7opBIC(); }
void ARM7opBICrrr() { ARM7addrDPrrr(); ARM7opBIC(); }

void ARM7opMVNreg() { ARM7addrDPreg(); ARM7opMVN(); }
void ARM7opMVNimm() { ARM7addrDPimm(); ARM7opMVN(); }
void ARM7opMVNlli() { ARM7addrDPlli(); ARM7opMVN(); }
void ARM7opMVNllr() { ARM7addrDPllr(); ARM7opMVN(); }
void ARM7opMVNlri() { ARM7addrDPlri(); ARM7opMVN(); }
void ARM7opMVNlrr() { ARM7addrDPlrr(); ARM7opMVN(); }
void ARM7opMVNari() { ARM7addrDPari(); ARM7opMVN(); }
void ARM7opMVNarr() { ARM7addrDParr(); ARM7opMVN(); }
void ARM7opMVNrri() { ARM7addrDPrri(); ARM7opMVN(); }
void ARM7opMVNrrr() { ARM7addrDPrrr(); ARM7opMVN(); }

void ARM7opANDSreg() { ARM7addrDPreg(); ARM7opAND(); ARM7addrDPflagBIT(); }
void ARM7opANDSimm() { ARM7addrDPimm(); ARM7opAND(); ARM7addrDPflagBIT(); }
void ARM7opANDSlli() { ARM7addrDPlli(); ARM7opAND(); ARM7addrDPflagBIT(); }
void ARM7opANDSllr() { ARM7addrDPllr(); ARM7opAND(); ARM7addrDPflagBIT(); }
void ARM7opANDSlri() { ARM7addrDPlri(); ARM7opAND(); ARM7addrDPflagBIT(); }
void ARM7opANDSlrr() { ARM7addrDPlrr(); ARM7opAND(); ARM7addrDPflagBIT(); }
void ARM7opANDSari() { ARM7addrDPari(); ARM7opAND(); ARM7addrDPflagBIT(); }
void ARM7opANDSarr() { ARM7addrDParr(); ARM7opAND(); ARM7addrDPflagBIT(); }
void ARM7opANDSrri() { ARM7addrDPrri(); ARM7opAND(); ARM7addrDPflagBIT(); }
void ARM7opANDSrrr() { ARM7addrDPrrr(); ARM7opAND(); ARM7addrDPflagBIT(); }

void ARM7opEORSreg() { ARM7addrDPreg(); ARM7opEOR(); ARM7addrDPflagBIT(); }
void ARM7opEORSimm() { ARM7addrDPimm(); ARM7opEOR(); ARM7addrDPflagBIT(); }
void ARM7opEORSlli() { ARM7addrDPlli(); ARM7opEOR(); ARM7addrDPflagBIT(); }
void ARM7opEORSllr() { ARM7addrDPllr(); ARM7opEOR(); ARM7addrDPflagBIT(); }
void ARM7opEORSlri() { ARM7addrDPlri(); ARM7opEOR(); ARM7addrDPflagBIT(); }
void ARM7opEORSlrr() { ARM7addrDPlrr(); ARM7opEOR(); ARM7addrDPflagBIT(); }
void ARM7opEORSari() { ARM7addrDPari(); ARM7opEOR(); ARM7addrDPflagBIT(); }
void ARM7opEORSarr() { ARM7addrDParr(); ARM7opEOR(); ARM7addrDPflagBIT(); }
void ARM7opEORSrri() { ARM7addrDPrri(); ARM7opEOR(); ARM7addrDPflagBIT(); }
void ARM7opEORSrrr() { ARM7addrDPrrr(); ARM7opEOR(); ARM7addrDPflagBIT(); }

void ARM7opSUBSreg() { ARM7addrDPreg(); ARM7opSUB(); ARM7addrDPflagSUB(); }
void ARM7opSUBSimm() { ARM7addrDPimm(); ARM7opSUB(); ARM7addrDPflagSUB(); }
void ARM7opSUBSlli() { ARM7addrDPlli(); ARM7opSUB(); ARM7addrDPflagSUB(); }
void ARM7opSUBSllr() { ARM7addrDPllr(); ARM7opSUB(); ARM7addrDPflagSUB(); }
void ARM7opSUBSlri() { ARM7addrDPlri(); ARM7opSUB(); ARM7addrDPflagSUB(); }
void ARM7opSUBSlrr() { ARM7addrDPlrr(); ARM7opSUB(); ARM7addrDPflagSUB(); }
void ARM7opSUBSari() { ARM7addrDPari(); ARM7opSUB(); ARM7addrDPflagSUB(); }
void ARM7opSUBSarr() { ARM7addrDParr(); ARM7opSUB(); ARM7addrDPflagSUB(); }
void ARM7opSUBSrri() { ARM7addrDPrri(); ARM7opSUB(); ARM7addrDPflagSUB(); }
void ARM7opSUBSrrr() { ARM7addrDPrrr(); ARM7opSUB(); ARM7addrDPflagSUB(); }

void ARM7opRSBSreg() { ARM7addrDPreg(); ARM7opRSB(); ARM7addrDPflagRSB(); }
void ARM7opRSBSimm() { ARM7addrDPimm(); ARM7opRSB(); ARM7addrDPflagRSB(); }
void ARM7opRSBSlli() { ARM7addrDPlli(); ARM7opRSB(); ARM7addrDPflagRSB(); }
void ARM7opRSBSllr() { ARM7addrDPllr(); ARM7opRSB(); ARM7addrDPflagRSB(); }
void ARM7opRSBSlri() { ARM7addrDPlri(); ARM7opRSB(); ARM7addrDPflagRSB(); }
void ARM7opRSBSlrr() { ARM7addrDPlrr(); ARM7opRSB(); ARM7addrDPflagRSB(); }
void ARM7opRSBSari() { ARM7addrDPari(); ARM7opRSB(); ARM7addrDPflagRSB(); }
void ARM7opRSBSarr() { ARM7addrDParr(); ARM7opRSB(); ARM7addrDPflagRSB(); }
void ARM7opRSBSrri() { ARM7addrDPrri(); ARM7opRSB(); ARM7addrDPflagRSB(); }
void ARM7opRSBSrrr() { ARM7addrDPrrr(); ARM7opRSB(); ARM7addrDPflagRSB(); }

void ARM7opADDSreg() { ARM7addrDPreg(); ARM7opADD(); ARM7addrDPflagADD(); }
void ARM7opADDSimm() { ARM7addrDPimm(); ARM7opADD(); ARM7addrDPflagADD(); }
void ARM7opADDSlli() { ARM7addrDPlli(); ARM7opADD(); ARM7addrDPflagADD(); }
void ARM7opADDSllr() { ARM7addrDPllr(); ARM7opADD(); ARM7addrDPflagADD(); }
void ARM7opADDSlri() { ARM7addrDPlri(); ARM7opADD(); ARM7addrDPflagADD(); }
void ARM7opADDSlrr() { ARM7addrDPlrr(); ARM7opADD(); ARM7addrDPflagADD(); }
void ARM7opADDSari() { ARM7addrDPari(); ARM7opADD(); ARM7addrDPflagADD(); }
void ARM7opADDSarr() { ARM7addrDParr(); ARM7opADD(); ARM7addrDPflagADD(); }
void ARM7opADDSrri() { ARM7addrDPrri(); ARM7opADD(); ARM7addrDPflagADD(); }
void ARM7opADDSrrr() { ARM7addrDPrrr(); ARM7opADD(); ARM7addrDPflagADD(); }

void ARM7opADCSreg() { ARM7addrDPreg(); ARM7opADC(); ARM7addrDPflagADC(); }
void ARM7opADCSimm() { ARM7addrDPimm(); ARM7opADC(); ARM7addrDPflagADC(); }
void ARM7opADCSlli() { ARM7addrDPlli(); ARM7opADC(); ARM7addrDPflagADC(); }
void ARM7opADCSllr() { ARM7addrDPllr(); ARM7opADC(); ARM7addrDPflagADC(); }
void ARM7opADCSlri() { ARM7addrDPlri(); ARM7opADC(); ARM7addrDPflagADC(); }
void ARM7opADCSlrr() { ARM7addrDPlrr(); ARM7opADC(); ARM7addrDPflagADC(); }
void ARM7opADCSari() { ARM7addrDPari(); ARM7opADC(); ARM7addrDPflagADC(); }
void ARM7opADCSarr() { ARM7addrDParr(); ARM7opADC(); ARM7addrDPflagADC(); }
void ARM7opADCSrri() { ARM7addrDPrri(); ARM7opADC(); ARM7addrDPflagADC(); }
void ARM7opADCSrrr() { ARM7addrDPrrr(); ARM7opADC(); ARM7addrDPflagADC(); }

void ARM7opSBCSreg() { ARM7addrDPreg(); ARM7opSBC(); ARM7addrDPflagSBC(); }
void ARM7opSBCSimm() { ARM7addrDPimm(); ARM7opSBC(); ARM7addrDPflagSBC(); }
void ARM7opSBCSlli() { ARM7addrDPlli(); ARM7opSBC(); ARM7addrDPflagSBC(); }
void ARM7opSBCSllr() { ARM7addrDPllr(); ARM7opSBC(); ARM7addrDPflagSBC(); }
void ARM7opSBCSlri() { ARM7addrDPlri(); ARM7opSBC(); ARM7addrDPflagSBC(); }
void ARM7opSBCSlrr() { ARM7addrDPlrr(); ARM7opSBC(); ARM7addrDPflagSBC(); }
void ARM7opSBCSari() { ARM7addrDPari(); ARM7opSBC(); ARM7addrDPflagSBC(); }
void ARM7opSBCSarr() { ARM7addrDParr(); ARM7opSBC(); ARM7addrDPflagSBC(); }
void ARM7opSBCSrri() { ARM7addrDPrri(); ARM7opSBC(); ARM7addrDPflagSBC(); }
void ARM7opSBCSrrr() { ARM7addrDPrrr(); ARM7opSBC(); ARM7addrDPflagSBC(); }

void ARM7opRSCSreg() { ARM7addrDPreg(); ARM7opRSC(); ARM7addrDPflagRSC(); }
void ARM7opRSCSimm() { ARM7addrDPimm(); ARM7opRSC(); ARM7addrDPflagRSC(); }
void ARM7opRSCSlli() { ARM7addrDPlli(); ARM7opRSC(); ARM7addrDPflagRSC(); }
void ARM7opRSCSllr() { ARM7addrDPllr(); ARM7opRSC(); ARM7addrDPflagRSC(); }
void ARM7opRSCSlri() { ARM7addrDPlri(); ARM7opRSC(); ARM7addrDPflagRSC(); }
void ARM7opRSCSlrr() { ARM7addrDPlrr(); ARM7opRSC(); ARM7addrDPflagRSC(); }
void ARM7opRSCSari() { ARM7addrDPari(); ARM7opRSC(); ARM7addrDPflagRSC(); }
void ARM7opRSCSarr() { ARM7addrDParr(); ARM7opRSC(); ARM7addrDPflagRSC(); }
void ARM7opRSCSrri() { ARM7addrDPrri(); ARM7opRSC(); ARM7addrDPflagRSC(); }
void ARM7opRSCSrrr() { ARM7addrDPrrr(); ARM7opRSC(); ARM7addrDPflagRSC(); }

void ARM7opTSTSreg() { ARM7addrDPreg(); ARM7opTST(); ARM7addrDPflagTST(); }
void ARM7opTSTSimm() { ARM7addrDPimm(); ARM7opTST(); ARM7addrDPflagTST(); }
void ARM7opTSTSlli() { ARM7addrDPlli(); ARM7opTST(); ARM7addrDPflagTST(); }
void ARM7opTSTSllr() { ARM7addrDPllr(); ARM7opTST(); ARM7addrDPflagTST(); }
void ARM7opTSTSlri() { ARM7addrDPlri(); ARM7opTST(); ARM7addrDPflagTST(); }
void ARM7opTSTSlrr() { ARM7addrDPlrr(); ARM7opTST(); ARM7addrDPflagTST(); }
void ARM7opTSTSari() { ARM7addrDPari(); ARM7opTST(); ARM7addrDPflagTST(); }
void ARM7opTSTSarr() { ARM7addrDParr(); ARM7opTST(); ARM7addrDPflagTST(); }
void ARM7opTSTSrri() { ARM7addrDPrri(); ARM7opTST(); ARM7addrDPflagTST(); }
void ARM7opTSTSrrr() { ARM7addrDPrrr(); ARM7opTST(); ARM7addrDPflagTST(); }

void ARM7opTEQSreg() { ARM7addrDPreg(); ARM7opTEQ(); ARM7addrDPflagTST(); }
void ARM7opTEQSimm() { ARM7addrDPimm(); ARM7opTEQ(); ARM7addrDPflagTST(); }
void ARM7opTEQSlli() { ARM7addrDPlli(); ARM7opTEQ(); ARM7addrDPflagTST(); }
void ARM7opTEQSllr() { ARM7addrDPllr(); ARM7opTEQ(); ARM7addrDPflagTST(); }
void ARM7opTEQSlri() { ARM7addrDPlri(); ARM7opTEQ(); ARM7addrDPflagTST(); }
void ARM7opTEQSlrr() { ARM7addrDPlrr(); ARM7opTEQ(); ARM7addrDPflagTST(); }
void ARM7opTEQSari() { ARM7addrDPari(); ARM7opTEQ(); ARM7addrDPflagTST(); }
void ARM7opTEQSarr() { ARM7addrDParr(); ARM7opTEQ(); ARM7addrDPflagTST(); }
void ARM7opTEQSrri() { ARM7addrDPrri(); ARM7opTEQ(); ARM7addrDPflagTST(); }
void ARM7opTEQSrrr() { ARM7addrDPrrr(); ARM7opTEQ(); ARM7addrDPflagTST(); }

void ARM7opCMPSreg() { ARM7addrDPreg(); ARM7opCMP(); ARM7addrDPflagCMP(); }
void ARM7opCMPSimm() { ARM7addrDPimm(); ARM7opCMP(); ARM7addrDPflagCMP(); }
void ARM7opCMPSlli() { ARM7addrDPlli(); ARM7opCMP(); ARM7addrDPflagCMP(); }
void ARM7opCMPSllr() { ARM7addrDPllr(); ARM7opCMP(); ARM7addrDPflagCMP(); }
void ARM7opCMPSlri() { ARM7addrDPlri(); ARM7opCMP(); ARM7addrDPflagCMP(); }
void ARM7opCMPSlrr() { ARM7addrDPlrr(); ARM7opCMP(); ARM7addrDPflagCMP(); }
void ARM7opCMPSari() { ARM7addrDPari(); ARM7opCMP(); ARM7addrDPflagCMP(); }
void ARM7opCMPSarr() { ARM7addrDParr(); ARM7opCMP(); ARM7addrDPflagCMP(); }
void ARM7opCMPSrri() { ARM7addrDPrri(); ARM7opCMP(); ARM7addrDPflagCMP(); }
void ARM7opCMPSrrr() { ARM7addrDPrrr(); ARM7opCMP(); ARM7addrDPflagCMP(); }

void ARM7opCMNSreg() { ARM7addrDPreg(); ARM7opCMN(); ARM7addrDPflagCMN(); }
void ARM7opCMNSimm() { ARM7addrDPimm(); ARM7opCMN(); ARM7addrDPflagCMN(); }
void ARM7opCMNSlli() { ARM7addrDPlli(); ARM7opCMN(); ARM7addrDPflagCMN(); }
void ARM7opCMNSllr() { ARM7addrDPllr(); ARM7opCMN(); ARM7addrDPflagCMN(); }
void ARM7opCMNSlri() { ARM7addrDPlri(); ARM7opCMN(); ARM7addrDPflagCMN(); }
void ARM7opCMNSlrr() { ARM7addrDPlrr(); ARM7opCMN(); ARM7addrDPflagCMN(); }
void ARM7opCMNSari() { ARM7addrDPari(); ARM7opCMN(); ARM7addrDPflagCMN(); }
void ARM7opCMNSarr() { ARM7addrDParr(); ARM7opCMN(); ARM7addrDPflagCMN(); }
void ARM7opCMNSrri() { ARM7addrDPrri(); ARM7opCMN(); ARM7addrDPflagCMN(); }
void ARM7opCMNSrrr() { ARM7addrDPrrr(); ARM7opCMN(); ARM7addrDPflagCMN(); }

void ARM7opORRSreg() { ARM7addrDPreg(); ARM7opORR(); ARM7addrDPflagBIT(); }
void ARM7opORRSimm() { ARM7addrDPimm(); ARM7opORR(); ARM7addrDPflagBIT(); }
void ARM7opORRSlli() { ARM7addrDPlli(); ARM7opORR(); ARM7addrDPflagBIT(); }
void ARM7opORRSllr() { ARM7addrDPllr(); ARM7opORR(); ARM7addrDPflagBIT(); }
void ARM7opORRSlri() { ARM7addrDPlri(); ARM7opORR(); ARM7addrDPflagBIT(); }
void ARM7opORRSlrr() { ARM7addrDPlrr(); ARM7opORR(); ARM7addrDPflagBIT(); }
void ARM7opORRSari() { ARM7addrDPari(); ARM7opORR(); ARM7addrDPflagBIT(); }
void ARM7opORRSarr() { ARM7addrDParr(); ARM7opORR(); ARM7addrDPflagBIT(); }
void ARM7opORRSrri() { ARM7addrDPrri(); ARM7opORR(); ARM7addrDPflagBIT(); }
void ARM7opORRSrrr() { ARM7addrDPrrr(); ARM7opORR(); ARM7addrDPflagBIT(); }

void ARM7opMOVSreg() { ARM7addrDPreg(); ARM7opMOV(); ARM7addrDPflagBIT(); }
void ARM7opMOVSimm() { ARM7addrDPimm(); ARM7opMOV(); ARM7addrDPflagBIT(); }
void ARM7opMOVSlli() { ARM7addrDPlli(); ARM7opMOV(); ARM7addrDPflagBIT(); }
void ARM7opMOVSllr() { ARM7addrDPllr(); ARM7opMOV(); ARM7addrDPflagBIT(); }
void ARM7opMOVSlri() { ARM7addrDPlri(); ARM7opMOV(); ARM7addrDPflagBIT(); }
void ARM7opMOVSlrr() { ARM7addrDPlrr(); ARM7opMOV(); ARM7addrDPflagBIT(); }
void ARM7opMOVSari() { ARM7addrDPari(); ARM7opMOV(); ARM7addrDPflagBIT(); }
void ARM7opMOVSarr() { ARM7addrDParr(); ARM7opMOV(); ARM7addrDPflagBIT(); }
void ARM7opMOVSrri() { ARM7addrDPrri(); ARM7opMOV(); ARM7addrDPflagBIT(); }
void ARM7opMOVSrrr() { ARM7addrDPrrr(); ARM7opMOV(); ARM7addrDPflagBIT(); }

void ARM7opBICSreg() { ARM7addrDPreg(); ARM7opBIC(); ARM7addrDPflagBIT(); }
void ARM7opBICSimm() { ARM7addrDPimm(); ARM7opBIC(); ARM7addrDPflagBIT(); }
void ARM7opBICSlli() { ARM7addrDPlli(); ARM7opBIC(); ARM7addrDPflagBIT(); }
void ARM7opBICSllr() { ARM7addrDPllr(); ARM7opBIC(); ARM7addrDPflagBIT(); }
void ARM7opBICSlri() { ARM7addrDPlri(); ARM7opBIC(); ARM7addrDPflagBIT(); }
void ARM7opBICSlrr() { ARM7addrDPlrr(); ARM7opBIC(); ARM7addrDPflagBIT(); }
void ARM7opBICSari() { ARM7addrDPari(); ARM7opBIC(); ARM7addrDPflagBIT(); }
void ARM7opBICSarr() { ARM7addrDParr(); ARM7opBIC(); ARM7addrDPflagBIT(); }
void ARM7opBICSrri() { ARM7addrDPrri(); ARM7opBIC(); ARM7addrDPflagBIT(); }
void ARM7opBICSrrr() { ARM7addrDPrrr(); ARM7opBIC(); ARM7addrDPflagBIT(); }

void ARM7opMVNSreg() { ARM7addrDPreg(); ARM7opMVN(); ARM7addrDPflagBIT(); }
void ARM7opMVNSimm() { ARM7addrDPimm(); ARM7opMVN(); ARM7addrDPflagBIT(); }
void ARM7opMVNSlli() { ARM7addrDPlli(); ARM7opMVN(); ARM7addrDPflagBIT(); }
void ARM7opMVNSllr() { ARM7addrDPllr(); ARM7opMVN(); ARM7addrDPflagBIT(); }
void ARM7opMVNSlri() { ARM7addrDPlri(); ARM7opMVN(); ARM7addrDPflagBIT(); }
void ARM7opMVNSlrr() { ARM7addrDPlrr(); ARM7opMVN(); ARM7addrDPflagBIT(); }
void ARM7opMVNSari() { ARM7addrDPari(); ARM7opMVN(); ARM7addrDPflagBIT(); }
void ARM7opMVNSarr() { ARM7addrDParr(); ARM7opMVN(); ARM7addrDPflagBIT(); }
void ARM7opMVNSrri() { ARM7addrDPrri(); ARM7opMVN(); ARM7addrDPflagBIT(); }
void ARM7opMVNSrrr() { ARM7addrDPrrr(); ARM7opMVN(); ARM7addrDPflagBIT(); }

//---Load/Store addressing modes, opcodes----------------------------------

#define ARM7addrLSofim() \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp3=arm7reg.curop&0x00000FFF; \
    arm7reg.tmp1=arm7reg.tmp2-arm7reg.tmp3

#define ARM7addrLSofip() \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp3=arm7reg.curop&0x00000FFF; \
    arm7reg.tmp1=arm7reg.tmp2+arm7reg.tmp3

#define ARM7addrLSlmofim() \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp3=((arm7reg.curop&0x00000F00)>>4)+(arm7reg.curop&0x0000000F); \
    arm7reg.tmp1=arm7reg.tmp2-arm7reg.tmp3

#define ARM7addrLSlmofip() \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp3=((arm7reg.curop&0x00000F00)>>4)+(arm7reg.curop&0x0000000F); \
    arm7reg.tmp1=arm7reg.tmp2+arm7reg.tmp3

#define ARM7addrLSofrm() \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2-arm7reg.r[ARM7OP_RM]

#define ARM7addrLSofrmll() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2-(arm7reg.r[ARM7OP_RM]<<arm7reg.tmp4)

#define ARM7addrLSofrmlr() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp4? \
                 (arm7reg.tmp2-(arm7reg.r[ARM7OP_RM]>>arm7reg.tmp4)): \
		 arm7reg.tmp2

#define ARM7addrLSofrmar() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp4? \
                 ((signed)(arm7reg.tmp2-(arm7reg.r[ARM7OP_RM]>>arm7reg.tmp4))): \
                 arm7reg.tmp2-((arm7reg.r[ARM7OP_RM]&0x80000000)?0xFFFFFFFF:0)

#define ARM7addrLSofrmrr() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp4? \
                 arm7reg.tmp2-((arm7reg.r[ARM7OP_RM]>>arm7reg.tmp4)| \
		     ((arm7reg.r[ARM7OP_RM]&((1<<arm7reg.tmp4)-1))<<(32-arm7reg.tmp4))): \
                 arm7reg.tmp2-((arm7reg.r[ARM7OP_RM]>>1)|((arm7reg.cpsr&ARMS_C)<<31)); \

#define ARM7addrLSofrp() \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2+arm7reg.r[ARM7OP_RM]

#define ARM7addrLSofrpll() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2+(arm7reg.r[ARM7OP_RM]<<arm7reg.tmp4)

#define ARM7addrLSofrplr() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp4? \
                 (arm7reg.tmp2+(arm7reg.r[ARM7OP_RM]>>arm7reg.tmp4)): \
		 arm7reg.tmp2

#define ARM7addrLSofrpar() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp4? \
                 ((signed)(arm7reg.tmp2+(arm7reg.r[ARM7OP_RM]>>arm7reg.tmp4))): \
                 arm7reg.tmp2+((arm7reg.r[ARM7OP_RM]&0x80000000)?0xFFFFFFFF:0)

#define ARM7addrLSofrprr() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp4? \
                 arm7reg.tmp2+((arm7reg.r[ARM7OP_RM]>>arm7reg.tmp4)| \
		     ((arm7reg.r[ARM7OP_RM]&((1<<arm7reg.tmp4)-1))<<(32-arm7reg.tmp4))): \
                 arm7reg.tmp2+((arm7reg.r[ARM7OP_RM]>>1)|((arm7reg.cpsr&ARMS_C)<<31))

#define ARM7addrLSprim() \
    arm7reg.tmp3=(arm7reg.curop&0x00000FFF); \
    arm7reg.r[ARM7OP_RN]-=arm7reg.tmp3; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2

#define ARM7addrLSprip() \
    arm7reg.tmp3=(arm7reg.curop&0x00000FFF); \
    arm7reg.r[ARM7OP_RN]+=arm7reg.tmp3; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2

#define ARM7addrLSlmprim() \
    arm7reg.tmp3=((arm7reg.curop&0x00000F00)>>4)+(arm7reg.curop&0x0000000F); \
    arm7reg.r[ARM7OP_RN]-=arm7reg.tmp3; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2

#define ARM7addrLSlmprip() \
    arm7reg.tmp3=((arm7reg.curop&0x00000F00)>>4)+(arm7reg.curop&0x0000000F); \
    arm7reg.r[ARM7OP_RN]+=arm7reg.tmp3; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2

#define ARM7addrLSprrm() \
    arm7reg.r[ARM7OP_RN]-=arm7reg.r[ARM7OP_RM]; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2

#define ARM7addrLSprrmll() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp3=(arm7reg.r[ARM7OP_RM]<<arm7reg.tmp4); \
    arm7reg.r[ARM7OP_RN]-=arm7reg.tmp3; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2

#define ARM7addrLSprrmlr() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp3=arm7reg.tmp4?(arm7reg.r[ARM7OP_RM]>>arm7reg.tmp4):0; \
    arm7reg.r[ARM7OP_RN]-=arm7reg.tmp3; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2

#define ARM7addrLSprrmar() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp3=arm7reg.tmp4? \
                 ((signed)(arm7reg.r[ARM7OP_RM]>>arm7reg.tmp4)): \
                 (arm7reg.r[ARM7OP_RM]&0x80000000)?0xFFFFFFFF:0; \
    arm7reg.r[ARM7OP_RN]-=arm7reg.tmp3; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2

#define ARM7addrLSprrmrr() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp3=arm7reg.tmp4? \
                 (arm7reg.r[ARM7OP_RM]>>arm7reg.tmp4)| \
		     ((arm7reg.r[ARM7OP_RM]&((1<<arm7reg.tmp4)-1))<<(32-arm7reg.tmp4)): \
                 (arm7reg.r[ARM7OP_RM]>>1)|((arm7reg.cpsr&ARMS_C)<<31); \
    arm7reg.r[ARM7OP_RN]-=arm7reg.tmp3; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2

#define ARM7addrLSprrp() \
    arm7reg.r[ARM7OP_RN]+=arm7reg.r[ARM7OP_RM]; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2

#define ARM7addrLSprrpll() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp3=(arm7reg.r[ARM7OP_RM]<<arm7reg.tmp4); \
    arm7reg.r[ARM7OP_RN]+=arm7reg.tmp3; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2

#define ARM7addrLSprrplr() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp3=arm7reg.tmp4?(arm7reg.r[ARM7OP_RM]>>arm7reg.tmp4):0; \
    arm7reg.r[ARM7OP_RN]+=arm7reg.tmp3; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2

#define ARM7addrLSprrpar() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp3=arm7reg.tmp4? \
                 ((signed)(arm7reg.r[ARM7OP_RM]>>arm7reg.tmp4)): \
                 (arm7reg.r[ARM7OP_RM]&0x80000000)?0xFFFFFFFF:0; \
    arm7reg.r[ARM7OP_RN]+=arm7reg.tmp3; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2

#define ARM7addrLSprrprr() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp3=arm7reg.tmp4? \
                 (arm7reg.r[ARM7OP_RM]>>arm7reg.tmp4)| \
		     ((arm7reg.r[ARM7OP_RM]&((1<<arm7reg.tmp4)-1))<<(32-arm7reg.tmp4)): \
                 (arm7reg.r[ARM7OP_RM]>>1)|((arm7reg.cpsr&ARMS_C)<<31); \
    arm7reg.r[ARM7OP_RN]+=arm7reg.tmp3; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2

#define ARM7addrLSptim() \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2; \
    arm7reg.tmp3=(arm7reg.curop&0x00000FFF); \
    arm7reg.r[ARM7OP_RN]-=arm7reg.tmp3

#define ARM7addrLSptip() \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2; \
    arm7reg.tmp3=(arm7reg.curop&0x00000FFF); \
    arm7reg.r[ARM7OP_RN]+=arm7reg.tmp3

#define ARM7addrLSlmptim() \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2; \
    arm7reg.tmp3=((arm7reg.curop&0x00000F00)>>4)+(arm7reg.curop&0x0000000F); \
    arm7reg.r[ARM7OP_RN]-=arm7reg.tmp3;

#define ARM7addrLSlmptip() \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2; \
    arm7reg.tmp3=((arm7reg.curop&0x00000F00)>>4)+(arm7reg.curop&0x0000000F); \
    arm7reg.r[ARM7OP_RN]+=arm7reg.tmp3;

#define ARM7addrLSptrm() \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2; \
    arm7reg.r[ARM7OP_RN]-=arm7reg.r[ARM7OP_RM]

#define ARM7addrLSptrmll() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp3=(arm7reg.r[ARM7OP_RM]<<arm7reg.tmp4); \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2; \
    arm7reg.r[ARM7OP_RN]-=arm7reg.tmp3

#define ARM7addrLSptrmlr() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp3=arm7reg.tmp4?(arm7reg.r[ARM7OP_RM]>>arm7reg.tmp4):0; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2; \
    arm7reg.r[ARM7OP_RN]-=arm7reg.tmp3

#define ARM7addrLSptrmar() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp3=arm7reg.tmp4? \
                 ((signed)(arm7reg.r[ARM7OP_RM]>>arm7reg.tmp4)): \
                 (arm7reg.r[ARM7OP_RM]&0x80000000)?0xFFFFFFFF:0; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2; \
    arm7reg.r[ARM7OP_RN]-=arm7reg.tmp3

#define ARM7addrLSptrmrr() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp3=arm7reg.tmp4? \
                 (arm7reg.r[ARM7OP_RM]>>arm7reg.tmp4)| \
		     ((arm7reg.r[ARM7OP_RM]&((1<<arm7reg.tmp4)-1))<<(32-arm7reg.tmp4)): \
                 (arm7reg.r[ARM7OP_RM]>>1)|((arm7reg.cpsr&ARMS_C)<<31); \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2; \
    arm7reg.r[ARM7OP_RN]-=arm7reg.tmp3

#define ARM7addrLSptrp() \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2; \
    arm7reg.r[ARM7OP_RN]+=arm7reg.r[ARM7OP_RM]

#define ARM7addrLSptrpll() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp3=(arm7reg.r[ARM7OP_RM]<<arm7reg.tmp4); \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2; \
    arm7reg.r[ARM7OP_RN]+=arm7reg.tmp3

#define ARM7addrLSptrplr() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp3=arm7reg.tmp4?(arm7reg.r[ARM7OP_RM]>>arm7reg.tmp4):0; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2; \
    arm7reg.r[ARM7OP_RN]+=arm7reg.tmp3

#define ARM7addrLSptrpar() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp3=arm7reg.tmp4? \
                 ((signed)(arm7reg.r[ARM7OP_RM]>>arm7reg.tmp4)): \
                 (arm7reg.r[ARM7OP_RM]&0x80000000)?0xFFFFFFFF:0; \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2; \
    arm7reg.r[ARM7OP_RN]+=arm7reg.tmp3

#define ARM7addrLSptrprr() \
    arm7reg.tmp4=(arm7reg.curop&0x00000F80)>>7; \
    arm7reg.tmp3=arm7reg.tmp4? \
                 (arm7reg.r[ARM7OP_RM]>>arm7reg.tmp4)| \
		     ((arm7reg.r[ARM7OP_RM]&((1<<arm7reg.tmp4)-1))<<(32-arm7reg.tmp4)): \
                 (arm7reg.r[ARM7OP_RM]>>1)|((arm7reg.cpsr&ARMS_C)<<31); \
    arm7reg.tmp2=(ARM7OP_RN==15)?arm7reg.r[ARM7OP_RN]+4:arm7reg.r[ARM7OP_RN]; \
    arm7reg.tmp1=arm7reg.tmp2; \
    arm7reg.r[ARM7OP_RN]+=arm7reg.tmp3

#define ARM7opLDR() \
    switch(arm7reg.tmp1&3) \
    { \
        case 0: arm7reg.tmp4=arm7reg.tmp1; break; \
	case 1: arm7reg.tmp4=(arm7reg.tmp1>> 8)|((arm7reg.tmp1&0x000000FF)<<24); break; \
	case 2: arm7reg.tmp4=(arm7reg.tmp1>>16)|((arm7reg.tmp1&0x0000FFFF)<<16); break; \
	case 3: arm7reg.tmp4=(arm7reg.tmp1>>24)|((arm7reg.tmp1&0x00FFFFFF)<< 8); break; \
    } \
    if(ARM7OP_RD==15) arm7reg.tmp4&=0xFFFFFFFC; \
    arm7reg.r[ARM7OP_RD]=MMUrdW(0,arm7reg.tmp4)

#define ARM7opLDRH() arm7reg.r[ARM7OP_RD]=0x00000000|MMUrdH(0,arm7reg.tmp1)
#define ARM7opLDRB() arm7reg.r[ARM7OP_RD]=0x00000000|MMUrdB(0,arm7reg.tmp1)

#define ARM7opLDRSH() \
    c=MMUrdW(0,arm7reg.tmp1); \
    arm7reg.r[ARM7OP_RD]=(c&0x8000)?(0xFFFF0000|c):(0x00000000|c)

#define ARM7opLDRSB() \
    c=MMUrdB(0,arm7reg.tmp1); \
    arm7reg.r[ARM7OP_RD]=(c&0x80)?(0xFFFFFF00|c):(0x00000000|c)

#define ARM7opLDRT() \
    switch(arm7reg.tmp1&3) \
    { \
        case 0: arm7reg.tmp4=arm7reg.tmp1; break; \
	case 1: arm7reg.tmp4=(arm7reg.tmp1>> 8)|((arm7reg.tmp1&0x000000FF)<<24); break; \
	case 2: arm7reg.tmp4=(arm7reg.tmp1>>16)|((arm7reg.tmp1&0x0000FFFF)<<16); break; \
	case 3: arm7reg.tmp4=(arm7reg.tmp1>>24)|((arm7reg.tmp1&0x00FFFFFF)<< 8); break; \
    } \
    arm7reg.r[ARM7OP_RD]=MMUrdW(0,arm7reg.tmp4)

#define ARM7opLDRBT() arm7reg.r[ARM7OP_RD]=0x00000000&MMUrdB(0,arm7reg.tmp1)
#define ARM7opSTR() MMUwrW(0,arm7reg.tmp1, arm7reg.r[ARM7OP_RD])
#define ARM7opSTRH() MMUwrH(0,arm7reg.tmp1, arm7reg.r[ARM7OP_RD]&0x0000FFFF)
#define ARM7opSTRB() MMUwrB(0,arm7reg.tmp1, arm7reg.r[ARM7OP_RD]&0x000000FF)
#define ARM7opSTRT() MMUwrW(0,arm7reg.tmp1, arm7reg.r[ARM7OP_RD])
#define ARM7opSTRBT() MMUwrB(0,arm7reg.tmp1, arm7reg.r[ARM7OP_RD]&0x000000FF)

void ARM7opLDRptim() { ARM7addrLSptim(); ARM7opLDR(); }
void ARM7opLDRptip() { ARM7addrLSptip(); ARM7opLDR(); }
void ARM7opLDRptrm() { ARM7addrLSptrm(); ARM7opLDR(); }
void ARM7opLDRptrp() { ARM7addrLSptrp(); ARM7opLDR(); }
void ARM7opLDRprim() { ARM7addrLSprim(); ARM7opLDR(); }
void ARM7opLDRprip() { ARM7addrLSprip(); ARM7opLDR(); }
void ARM7opLDRprrm() { ARM7addrLSprrm(); ARM7opLDR(); }
void ARM7opLDRprrp() { ARM7addrLSprrp(); ARM7opLDR(); }
void ARM7opLDRofim() { ARM7addrLSofim(); ARM7opLDR(); }
void ARM7opLDRofip() { ARM7addrLSofip(); ARM7opLDR(); }
void ARM7opLDRofrm() { ARM7addrLSofrm(); ARM7opLDR(); }
void ARM7opLDRofrp() { ARM7addrLSofrp(); ARM7opLDR(); }

void ARM7opSTRptim() { ARM7addrLSptim(); ARM7opSTR(); }
void ARM7opSTRptip() { ARM7addrLSptip(); ARM7opSTR(); }
void ARM7opSTRptrm() { ARM7addrLSptrm(); ARM7opSTR(); }
void ARM7opSTRptrp() { ARM7addrLSptrp(); ARM7opSTR(); }
void ARM7opSTRprim() { ARM7addrLSprim(); ARM7opSTR(); }
void ARM7opSTRprip() { ARM7addrLSprip(); ARM7opSTR(); }
void ARM7opSTRprrm() { ARM7addrLSprrm(); ARM7opSTR(); }
void ARM7opSTRprrp() { ARM7addrLSprrp(); ARM7opSTR(); }
void ARM7opSTRofim() { ARM7addrLSofim(); ARM7opSTR(); }
void ARM7opSTRofip() { ARM7addrLSofip(); ARM7opSTR(); }
void ARM7opSTRofrm() { ARM7addrLSofrm(); ARM7opSTR(); }
void ARM7opSTRofrp() { ARM7addrLSofrp(); ARM7opSTR(); }

void ARM7opLDRBptim() { ARM7addrLSptim(); ARM7opLDRB(); }
void ARM7opLDRBptip() { ARM7addrLSptip(); ARM7opLDRB(); }
void ARM7opLDRBptrm() { ARM7addrLSptrm(); ARM7opLDRB(); }
void ARM7opLDRBptrp() { ARM7addrLSptrp(); ARM7opLDRB(); }
void ARM7opLDRBprim() { ARM7addrLSprim(); ARM7opLDRB(); }
void ARM7opLDRBprip() { ARM7addrLSprip(); ARM7opLDRB(); }
void ARM7opLDRBprrm() { ARM7addrLSprrm(); ARM7opLDRB(); }
void ARM7opLDRBprrp() { ARM7addrLSprrp(); ARM7opLDRB(); }
void ARM7opLDRBofim() { ARM7addrLSofim(); ARM7opLDRB(); }
void ARM7opLDRBofip() { ARM7addrLSofip(); ARM7opLDRB(); }
void ARM7opLDRBofrm() { ARM7addrLSofrm(); ARM7opLDRB(); }
void ARM7opLDRBofrp() { ARM7addrLSofrp(); ARM7opLDRB(); }

void ARM7opSTRBptim() { ARM7addrLSptim(); ARM7opSTRB(); }
void ARM7opSTRBptip() { ARM7addrLSptip(); ARM7opSTRB(); }
void ARM7opSTRBptrm() { ARM7addrLSptrm(); ARM7opSTRB(); }
void ARM7opSTRBptrp() { ARM7addrLSptrp(); ARM7opSTRB(); }
void ARM7opSTRBprim() { ARM7addrLSprim(); ARM7opSTRB(); }
void ARM7opSTRBprip() { ARM7addrLSprip(); ARM7opSTRB(); }
void ARM7opSTRBprrm() { ARM7addrLSprrm(); ARM7opSTRB(); }
void ARM7opSTRBprrp() { ARM7addrLSprrp(); ARM7opSTRB(); }
void ARM7opSTRBofim() { ARM7addrLSofim(); ARM7opSTRB(); }
void ARM7opSTRBofip() { ARM7addrLSofip(); ARM7opSTRB(); }
void ARM7opSTRBofrm() { ARM7addrLSofrm(); ARM7opSTRB(); }
void ARM7opSTRBofrp() { ARM7addrLSofrp(); ARM7opSTRB(); }

void ARM7opLDRHptim() { ARM7addrLSlmptim(); ARM7opLDRH(); }
void ARM7opLDRHptip() { ARM7addrLSlmptip(); ARM7opLDRH(); }
void ARM7opLDRHptrm() { ARM7addrLSptrm(); ARM7opLDRH(); }
void ARM7opLDRHptrp() { ARM7addrLSptrp(); ARM7opLDRH(); }
void ARM7opLDRHprim() { ARM7addrLSlmprim(); ARM7opLDRH(); }
void ARM7opLDRHprip() { ARM7addrLSlmprip(); ARM7opLDRH(); }
void ARM7opLDRHprrm() { ARM7addrLSprrm(); ARM7opLDRH(); }
void ARM7opLDRHprrp() { ARM7addrLSprrp(); ARM7opLDRH(); }
void ARM7opLDRHofim() { ARM7addrLSlmofim(); ARM7opLDRH(); }
void ARM7opLDRHofip() { ARM7addrLSlmofip(); ARM7opLDRH(); }
void ARM7opLDRHofrm() { ARM7addrLSofrm(); ARM7opLDRH(); }
void ARM7opLDRHofrp() { ARM7addrLSofrp(); ARM7opLDRH(); }

void ARM7opSTRHptim() { ARM7addrLSlmptim(); ARM7opSTRH(); }
void ARM7opSTRHptip() { ARM7addrLSlmptip(); ARM7opSTRH(); }
void ARM7opSTRHptrm() { ARM7addrLSptrm(); ARM7opSTRH(); }
void ARM7opSTRHptrp() { ARM7addrLSptrp(); ARM7opSTRH(); }
void ARM7opSTRHprim() { ARM7addrLSlmprim(); ARM7opSTRH(); }
void ARM7opSTRHprip() { ARM7addrLSlmprip(); ARM7opSTRH(); }
void ARM7opSTRHprrm() { ARM7addrLSprrm(); ARM7opSTRH(); }
void ARM7opSTRHprrp() { ARM7addrLSprrp(); ARM7opSTRH(); }
void ARM7opSTRHofim() { ARM7addrLSlmofim(); ARM7opSTRH(); }
void ARM7opSTRHofip() { ARM7addrLSlmofip(); ARM7opSTRH(); }
void ARM7opSTRHofrm() { ARM7addrLSofrm(); ARM7opSTRH(); }
void ARM7opSTRHofrp() { ARM7addrLSofrp(); ARM7opSTRH(); }

void ARM7opLDRTptim() { ARM7addrLSptim(); ARM7opLDRT(); }
void ARM7opLDRTptip() { ARM7addrLSptip(); ARM7opLDRT(); }
void ARM7opLDRTptrm() { ARM7addrLSptrm(); ARM7opLDRT(); }
void ARM7opLDRTptrp() { ARM7addrLSptrp(); ARM7opLDRT(); }
void ARM7opLDRTprim() { ARM7addrLSprim(); ARM7opLDRT(); }
void ARM7opLDRTprip() { ARM7addrLSprip(); ARM7opLDRT(); }
void ARM7opLDRTprrm() { ARM7addrLSprrm(); ARM7opLDRT(); }
void ARM7opLDRTprrp() { ARM7addrLSprrp(); ARM7opLDRT(); }
void ARM7opLDRTofim() { ARM7addrLSofim(); ARM7opLDRT(); }
void ARM7opLDRTofip() { ARM7addrLSofip(); ARM7opLDRT(); }
void ARM7opLDRTofrm() { ARM7addrLSofrm(); ARM7opLDRT(); }
void ARM7opLDRTofrp() { ARM7addrLSofrp(); ARM7opLDRT(); }

void ARM7opSTRTptim() { ARM7addrLSptim(); ARM7opSTRT(); }
void ARM7opSTRTptip() { ARM7addrLSptip(); ARM7opSTRT(); }
void ARM7opSTRTptrm() { ARM7addrLSptrm(); ARM7opSTRT(); }
void ARM7opSTRTptrp() { ARM7addrLSptrp(); ARM7opSTRT(); }
void ARM7opSTRTprim() { ARM7addrLSprim(); ARM7opSTRT(); }
void ARM7opSTRTprip() { ARM7addrLSprip(); ARM7opSTRT(); }
void ARM7opSTRTprrm() { ARM7addrLSprrm(); ARM7opSTRT(); }
void ARM7opSTRTprrp() { ARM7addrLSprrp(); ARM7opSTRT(); }
void ARM7opSTRTofim() { ARM7addrLSofim(); ARM7opSTRT(); }
void ARM7opSTRTofip() { ARM7addrLSofip(); ARM7opSTRT(); }
void ARM7opSTRTofrm() { ARM7addrLSofrm(); ARM7opSTRT(); }
void ARM7opSTRTofrp() { ARM7addrLSofrp(); ARM7opSTRT(); }

void ARM7opLDRBTptim() { ARM7addrLSptim(); ARM7opLDRBT(); }
void ARM7opLDRBTptip() { ARM7addrLSptip(); ARM7opLDRBT(); }
void ARM7opLDRBTptrm() { ARM7addrLSptrm(); ARM7opLDRBT(); }
void ARM7opLDRBTptrp() { ARM7addrLSptrp(); ARM7opLDRBT(); }
void ARM7opLDRBTprim() { ARM7addrLSprim(); ARM7opLDRBT(); }
void ARM7opLDRBTprip() { ARM7addrLSprip(); ARM7opLDRBT(); }
void ARM7opLDRBTprrm() { ARM7addrLSprrm(); ARM7opLDRBT(); }
void ARM7opLDRBTprrp() { ARM7addrLSprrp(); ARM7opLDRBT(); }
void ARM7opLDRBTofim() { ARM7addrLSofim(); ARM7opLDRBT(); }
void ARM7opLDRBTofip() { ARM7addrLSofip(); ARM7opLDRBT(); }
void ARM7opLDRBTofrm() { ARM7addrLSofrm(); ARM7opLDRBT(); }
void ARM7opLDRBTofrp() { ARM7addrLSofrp(); ARM7opLDRBT(); }

void ARM7opSTRBTptim() { ARM7addrLSptim(); ARM7opSTRBT(); }
void ARM7opSTRBTptip() { ARM7addrLSptip(); ARM7opSTRBT(); }
void ARM7opSTRBTptrm() { ARM7addrLSptrm(); ARM7opSTRBT(); }
void ARM7opSTRBTptrp() { ARM7addrLSptrp(); ARM7opSTRBT(); }
void ARM7opSTRBTprim() { ARM7addrLSprim(); ARM7opSTRBT(); }
void ARM7opSTRBTprip() { ARM7addrLSprip(); ARM7opSTRBT(); }
void ARM7opSTRBTprrm() { ARM7addrLSprrm(); ARM7opSTRBT(); }
void ARM7opSTRBTprrp() { ARM7addrLSprrp(); ARM7opSTRBT(); }
void ARM7opSTRBTofim() { ARM7addrLSofim(); ARM7opSTRBT(); }
void ARM7opSTRBTofip() { ARM7addrLSofip(); ARM7opSTRBT(); }
void ARM7opSTRBTofrm() { ARM7addrLSofrm(); ARM7opSTRBT(); }
void ARM7opSTRBTofrp() { ARM7addrLSofrp(); ARM7opSTRBT(); }

void ARM7opLDRSBptim() { u8 c; ARM7addrLSlmptim(); ARM7opLDRSB(); }
void ARM7opLDRSBptip() { u8 c; ARM7addrLSlmptip(); ARM7opLDRSB(); }
void ARM7opLDRSBptrm() { u8 c; ARM7addrLSptrm(); ARM7opLDRSB(); }
void ARM7opLDRSBptrp() { u8 c; ARM7addrLSptrp(); ARM7opLDRSB(); }
void ARM7opLDRSBprim() { u8 c; ARM7addrLSlmprim(); ARM7opLDRSB(); }
void ARM7opLDRSBprip() { u8 c; ARM7addrLSlmprip(); ARM7opLDRSB(); }
void ARM7opLDRSBprrm() { u8 c; ARM7addrLSprrm(); ARM7opLDRSB(); }
void ARM7opLDRSBprrp() { u8 c; ARM7addrLSprrp(); ARM7opLDRSB(); }
void ARM7opLDRSBofim() { u8 c; ARM7addrLSlmofim(); ARM7opLDRSB(); }
void ARM7opLDRSBofip() { u8 c; ARM7addrLSlmofip(); ARM7opLDRSB(); }
void ARM7opLDRSBofrm() { u8 c; ARM7addrLSofrm(); ARM7opLDRSB(); }
void ARM7opLDRSBofrp() { u8 c; ARM7addrLSofrp(); ARM7opLDRSB(); }

void ARM7opLDRSHptim() { u16 c; ARM7addrLSlmptim(); ARM7opLDRSH(); }
void ARM7opLDRSHptip() { u16 c; ARM7addrLSlmptip(); ARM7opLDRSH(); }
void ARM7opLDRSHptrm() { u16 c; ARM7addrLSptrm(); ARM7opLDRSH(); }
void ARM7opLDRSHptrp() { u16 c; ARM7addrLSptrp(); ARM7opLDRSH(); }
void ARM7opLDRSHprim() { u16 c; ARM7addrLSlmprim(); ARM7opLDRSH(); }
void ARM7opLDRSHprip() { u16 c; ARM7addrLSlmprip(); ARM7opLDRSH(); }
void ARM7opLDRSHprrm() { u16 c; ARM7addrLSprrm(); ARM7opLDRSH(); }
void ARM7opLDRSHprrp() { u16 c; ARM7addrLSprrp(); ARM7opLDRSH(); }
void ARM7opLDRSHofim() { u16 c; ARM7addrLSlmofim(); ARM7opLDRSH(); }
void ARM7opLDRSHofip() { u16 c; ARM7addrLSlmofip(); ARM7opLDRSH(); }
void ARM7opLDRSHofrm() { u16 c; ARM7addrLSofrm(); ARM7opLDRSH(); }
void ARM7opLDRSHofrp() { u16 c; ARM7addrLSofrp(); ARM7opLDRSH(); }

void ARM7opLDRptrmll() { ARM7addrLSptrmll(); ARM7opLDR(); }
void ARM7opLDRptrmlr() { ARM7addrLSptrmlr(); ARM7opLDR(); }
void ARM7opLDRptrmar() { ARM7addrLSptrmar(); ARM7opLDR(); }
void ARM7opLDRptrmrr() { ARM7addrLSptrmrr(); ARM7opLDR(); }
void ARM7opLDRptrpll() { ARM7addrLSptrpll(); ARM7opLDR(); }
void ARM7opLDRptrplr() { ARM7addrLSptrplr(); ARM7opLDR(); }
void ARM7opLDRptrpar() { ARM7addrLSptrpar(); ARM7opLDR(); }
void ARM7opLDRptrprr() { ARM7addrLSptrprr(); ARM7opLDR(); }
void ARM7opLDRprrmll() { ARM7addrLSprrmll(); ARM7opLDR(); }
void ARM7opLDRprrmlr() { ARM7addrLSprrmlr(); ARM7opLDR(); }
void ARM7opLDRprrmar() { ARM7addrLSprrmar(); ARM7opLDR(); }
void ARM7opLDRprrmrr() { ARM7addrLSprrmrr(); ARM7opLDR(); }
void ARM7opLDRprrpll() { ARM7addrLSprrpll(); ARM7opLDR(); }
void ARM7opLDRprrplr() { ARM7addrLSprrplr(); ARM7opLDR(); }
void ARM7opLDRprrpar() { ARM7addrLSprrpar(); ARM7opLDR(); }
void ARM7opLDRprrprr() { ARM7addrLSprrprr(); ARM7opLDR(); }
void ARM7opLDRofrmll() { ARM7addrLSofrmll(); ARM7opLDR(); }
void ARM7opLDRofrmlr() { ARM7addrLSofrmlr(); ARM7opLDR(); }
void ARM7opLDRofrmar() { ARM7addrLSofrmar(); ARM7opLDR(); }
void ARM7opLDRofrmrr() { ARM7addrLSofrmrr(); ARM7opLDR(); }
void ARM7opLDRofrpll() { ARM7addrLSofrpll(); ARM7opLDR(); }
void ARM7opLDRofrplr() { ARM7addrLSofrplr(); ARM7opLDR(); }
void ARM7opLDRofrpar() { ARM7addrLSofrpar(); ARM7opLDR(); }
void ARM7opLDRofrprr() { ARM7addrLSofrprr(); ARM7opLDR(); }

void ARM7opSTRptrmll() { ARM7addrLSptrmll(); ARM7opSTR(); }
void ARM7opSTRptrmlr() { ARM7addrLSptrmlr(); ARM7opSTR(); }
void ARM7opSTRptrmar() { ARM7addrLSptrmar(); ARM7opSTR(); }
void ARM7opSTRptrmrr() { ARM7addrLSptrmrr(); ARM7opSTR(); }
void ARM7opSTRptrpll() { ARM7addrLSptrpll(); ARM7opSTR(); }
void ARM7opSTRptrplr() { ARM7addrLSptrplr(); ARM7opSTR(); }
void ARM7opSTRptrpar() { ARM7addrLSptrpar(); ARM7opSTR(); }
void ARM7opSTRptrprr() { ARM7addrLSptrprr(); ARM7opSTR(); }
void ARM7opSTRprrmll() { ARM7addrLSprrmll(); ARM7opSTR(); }
void ARM7opSTRprrmlr() { ARM7addrLSprrmlr(); ARM7opSTR(); }
void ARM7opSTRprrmar() { ARM7addrLSprrmar(); ARM7opSTR(); }
void ARM7opSTRprrmrr() { ARM7addrLSprrmrr(); ARM7opSTR(); }
void ARM7opSTRprrpll() { ARM7addrLSprrpll(); ARM7opSTR(); }
void ARM7opSTRprrplr() { ARM7addrLSprrplr(); ARM7opSTR(); }
void ARM7opSTRprrpar() { ARM7addrLSprrpar(); ARM7opSTR(); }
void ARM7opSTRprrprr() { ARM7addrLSprrprr(); ARM7opSTR(); }
void ARM7opSTRofrmll() { ARM7addrLSofrmll(); ARM7opSTR(); }
void ARM7opSTRofrmlr() { ARM7addrLSofrmlr(); ARM7opSTR(); }
void ARM7opSTRofrmar() { ARM7addrLSofrmar(); ARM7opSTR(); }
void ARM7opSTRofrmrr() { ARM7addrLSofrmrr(); ARM7opSTR(); }
void ARM7opSTRofrpll() { ARM7addrLSofrpll(); ARM7opSTR(); }
void ARM7opSTRofrplr() { ARM7addrLSofrplr(); ARM7opSTR(); }
void ARM7opSTRofrpar() { ARM7addrLSofrpar(); ARM7opSTR(); }
void ARM7opSTRofrprr() { ARM7addrLSofrprr(); ARM7opSTR(); }

void ARM7opLDRBptrmll() { ARM7addrLSptrmll(); ARM7opLDRB(); }
void ARM7opLDRBptrmlr() { ARM7addrLSptrmlr(); ARM7opLDRB(); }
void ARM7opLDRBptrmar() { ARM7addrLSptrmar(); ARM7opLDRB(); }
void ARM7opLDRBptrmrr() { ARM7addrLSptrmrr(); ARM7opLDRB(); }
void ARM7opLDRBptrpll() { ARM7addrLSptrpll(); ARM7opLDRB(); }
void ARM7opLDRBptrplr() { ARM7addrLSptrplr(); ARM7opLDRB(); }
void ARM7opLDRBptrpar() { ARM7addrLSptrpar(); ARM7opLDRB(); }
void ARM7opLDRBptrprr() { ARM7addrLSptrprr(); ARM7opLDRB(); }
void ARM7opLDRBprrmll() { ARM7addrLSprrmll(); ARM7opLDRB(); }
void ARM7opLDRBprrmlr() { ARM7addrLSprrmlr(); ARM7opLDRB(); }
void ARM7opLDRBprrmar() { ARM7addrLSprrmar(); ARM7opLDRB(); }
void ARM7opLDRBprrmrr() { ARM7addrLSprrmrr(); ARM7opLDRB(); }
void ARM7opLDRBprrpll() { ARM7addrLSprrpll(); ARM7opLDRB(); }
void ARM7opLDRBprrplr() { ARM7addrLSprrplr(); ARM7opLDRB(); }
void ARM7opLDRBprrpar() { ARM7addrLSprrpar(); ARM7opLDRB(); }
void ARM7opLDRBprrprr() { ARM7addrLSprrprr(); ARM7opLDRB(); }
void ARM7opLDRBofrmll() { ARM7addrLSofrmll(); ARM7opLDRB(); }
void ARM7opLDRBofrmlr() { ARM7addrLSofrmlr(); ARM7opLDRB(); }
void ARM7opLDRBofrmar() { ARM7addrLSofrmar(); ARM7opLDRB(); }
void ARM7opLDRBofrmrr() { ARM7addrLSofrmrr(); ARM7opLDRB(); }
void ARM7opLDRBofrpll() { ARM7addrLSofrpll(); ARM7opLDRB(); }
void ARM7opLDRBofrplr() { ARM7addrLSofrplr(); ARM7opLDRB(); }
void ARM7opLDRBofrpar() { ARM7addrLSofrpar(); ARM7opLDRB(); }
void ARM7opLDRBofrprr() { ARM7addrLSofrprr(); ARM7opLDRB(); }

void ARM7opSTRBptrmll() { ARM7addrLSptrmll(); ARM7opSTRB(); }
void ARM7opSTRBptrmlr() { ARM7addrLSptrmlr(); ARM7opSTRB(); }
void ARM7opSTRBptrmar() { ARM7addrLSptrmar(); ARM7opSTRB(); }
void ARM7opSTRBptrmrr() { ARM7addrLSptrmrr(); ARM7opSTRB(); }
void ARM7opSTRBptrpll() { ARM7addrLSptrpll(); ARM7opSTRB(); }
void ARM7opSTRBptrplr() { ARM7addrLSptrplr(); ARM7opSTRB(); }
void ARM7opSTRBptrpar() { ARM7addrLSptrpar(); ARM7opSTRB(); }
void ARM7opSTRBptrprr() { ARM7addrLSptrprr(); ARM7opSTRB(); }
void ARM7opSTRBprrmll() { ARM7addrLSprrmll(); ARM7opSTRB(); }
void ARM7opSTRBprrmlr() { ARM7addrLSprrmlr(); ARM7opSTRB(); }
void ARM7opSTRBprrmar() { ARM7addrLSprrmar(); ARM7opSTRB(); }
void ARM7opSTRBprrmrr() { ARM7addrLSprrmrr(); ARM7opSTRB(); }
void ARM7opSTRBprrpll() { ARM7addrLSprrpll(); ARM7opSTRB(); }
void ARM7opSTRBprrplr() { ARM7addrLSprrplr(); ARM7opSTRB(); }
void ARM7opSTRBprrpar() { ARM7addrLSprrpar(); ARM7opSTRB(); }
void ARM7opSTRBprrprr() { ARM7addrLSprrprr(); ARM7opSTRB(); }
void ARM7opSTRBofrmll() { ARM7addrLSofrmll(); ARM7opSTRB(); }
void ARM7opSTRBofrmlr() { ARM7addrLSofrmlr(); ARM7opSTRB(); }
void ARM7opSTRBofrmar() { ARM7addrLSofrmar(); ARM7opSTRB(); }
void ARM7opSTRBofrmrr() { ARM7addrLSofrmrr(); ARM7opSTRB(); }
void ARM7opSTRBofrpll() { ARM7addrLSofrpll(); ARM7opSTRB(); }
void ARM7opSTRBofrplr() { ARM7addrLSofrplr(); ARM7opSTRB(); }
void ARM7opSTRBofrpar() { ARM7addrLSofrpar(); ARM7opSTRB(); }
void ARM7opSTRBofrprr() { ARM7addrLSofrprr(); ARM7opSTRB(); }

void ARM7opLDRTptrmll() { ARM7addrLSptrmll(); ARM7opLDRT(); }
void ARM7opLDRTptrmlr() { ARM7addrLSptrmlr(); ARM7opLDRT(); }
void ARM7opLDRTptrmar() { ARM7addrLSptrmar(); ARM7opLDRT(); }
void ARM7opLDRTptrmrr() { ARM7addrLSptrmrr(); ARM7opLDRT(); }
void ARM7opLDRTptrpll() { ARM7addrLSptrpll(); ARM7opLDRT(); }
void ARM7opLDRTptrplr() { ARM7addrLSptrplr(); ARM7opLDRT(); }
void ARM7opLDRTptrpar() { ARM7addrLSptrpar(); ARM7opLDRT(); }
void ARM7opLDRTptrprr() { ARM7addrLSptrprr(); ARM7opLDRT(); }
void ARM7opLDRTprrmll() { ARM7addrLSprrmll(); ARM7opLDRT(); }
void ARM7opLDRTprrmlr() { ARM7addrLSprrmlr(); ARM7opLDRT(); }
void ARM7opLDRTprrmar() { ARM7addrLSprrmar(); ARM7opLDRT(); }
void ARM7opLDRTprrmrr() { ARM7addrLSprrmrr(); ARM7opLDRT(); }
void ARM7opLDRTprrpll() { ARM7addrLSprrpll(); ARM7opLDRT(); }
void ARM7opLDRTprrplr() { ARM7addrLSprrplr(); ARM7opLDRT(); }
void ARM7opLDRTprrpar() { ARM7addrLSprrpar(); ARM7opLDRT(); }
void ARM7opLDRTprrprr() { ARM7addrLSprrprr(); ARM7opLDRT(); }
void ARM7opLDRTofrmll() { ARM7addrLSofrmll(); ARM7opLDRT(); }
void ARM7opLDRTofrmlr() { ARM7addrLSofrmlr(); ARM7opLDRT(); }
void ARM7opLDRTofrmar() { ARM7addrLSofrmar(); ARM7opLDRT(); }
void ARM7opLDRTofrmrr() { ARM7addrLSofrmrr(); ARM7opLDRT(); }
void ARM7opLDRTofrpll() { ARM7addrLSofrpll(); ARM7opLDRT(); }
void ARM7opLDRTofrplr() { ARM7addrLSofrplr(); ARM7opLDRT(); }
void ARM7opLDRTofrpar() { ARM7addrLSofrpar(); ARM7opLDRT(); }
void ARM7opLDRTofrprr() { ARM7addrLSofrprr(); ARM7opLDRT(); }

void ARM7opSTRTptrmll() { ARM7addrLSptrmll(); ARM7opSTRT(); }
void ARM7opSTRTptrmlr() { ARM7addrLSptrmlr(); ARM7opSTRT(); }
void ARM7opSTRTptrmar() { ARM7addrLSptrmar(); ARM7opSTRT(); }
void ARM7opSTRTptrmrr() { ARM7addrLSptrmrr(); ARM7opSTRT(); }
void ARM7opSTRTptrpll() { ARM7addrLSptrpll(); ARM7opSTRT(); }
void ARM7opSTRTptrplr() { ARM7addrLSptrplr(); ARM7opSTRT(); }
void ARM7opSTRTptrpar() { ARM7addrLSptrpar(); ARM7opSTRT(); }
void ARM7opSTRTptrprr() { ARM7addrLSptrprr(); ARM7opSTRT(); }
void ARM7opSTRTprrmll() { ARM7addrLSprrmll(); ARM7opSTRT(); }
void ARM7opSTRTprrmlr() { ARM7addrLSprrmlr(); ARM7opSTRT(); }
void ARM7opSTRTprrmar() { ARM7addrLSprrmar(); ARM7opSTRT(); }
void ARM7opSTRTprrmrr() { ARM7addrLSprrmrr(); ARM7opSTRT(); }
void ARM7opSTRTprrpll() { ARM7addrLSprrpll(); ARM7opSTRT(); }
void ARM7opSTRTprrplr() { ARM7addrLSprrplr(); ARM7opSTRT(); }
void ARM7opSTRTprrpar() { ARM7addrLSprrpar(); ARM7opSTRT(); }
void ARM7opSTRTprrprr() { ARM7addrLSprrprr(); ARM7opSTRT(); }
void ARM7opSTRTofrmll() { ARM7addrLSofrmll(); ARM7opSTRT(); }
void ARM7opSTRTofrmlr() { ARM7addrLSofrmlr(); ARM7opSTRT(); }
void ARM7opSTRTofrmar() { ARM7addrLSofrmar(); ARM7opSTRT(); }
void ARM7opSTRTofrmrr() { ARM7addrLSofrmrr(); ARM7opSTRT(); }
void ARM7opSTRTofrpll() { ARM7addrLSofrpll(); ARM7opSTRT(); }
void ARM7opSTRTofrplr() { ARM7addrLSofrplr(); ARM7opSTRT(); }
void ARM7opSTRTofrpar() { ARM7addrLSofrpar(); ARM7opSTRT(); }
void ARM7opSTRTofrprr() { ARM7addrLSofrprr(); ARM7opSTRT(); }

void ARM7opLDRBTptrmll() { ARM7addrLSptrmll(); ARM7opLDRBT(); }
void ARM7opLDRBTptrmlr() { ARM7addrLSptrmlr(); ARM7opLDRBT(); }
void ARM7opLDRBTptrmar() { ARM7addrLSptrmar(); ARM7opLDRBT(); }
void ARM7opLDRBTptrmrr() { ARM7addrLSptrmrr(); ARM7opLDRBT(); }
void ARM7opLDRBTptrpll() { ARM7addrLSptrpll(); ARM7opLDRBT(); }
void ARM7opLDRBTptrplr() { ARM7addrLSptrplr(); ARM7opLDRBT(); }
void ARM7opLDRBTptrpar() { ARM7addrLSptrpar(); ARM7opLDRBT(); }
void ARM7opLDRBTptrprr() { ARM7addrLSptrprr(); ARM7opLDRBT(); }
void ARM7opLDRBTprrmll() { ARM7addrLSprrmll(); ARM7opLDRBT(); }
void ARM7opLDRBTprrmlr() { ARM7addrLSprrmlr(); ARM7opLDRBT(); }
void ARM7opLDRBTprrmar() { ARM7addrLSprrmar(); ARM7opLDRBT(); }
void ARM7opLDRBTprrmrr() { ARM7addrLSprrmrr(); ARM7opLDRBT(); }
void ARM7opLDRBTprrpll() { ARM7addrLSprrpll(); ARM7opLDRBT(); }
void ARM7opLDRBTprrplr() { ARM7addrLSprrplr(); ARM7opLDRBT(); }
void ARM7opLDRBTprrpar() { ARM7addrLSprrpar(); ARM7opLDRBT(); }
void ARM7opLDRBTprrprr() { ARM7addrLSprrprr(); ARM7opLDRBT(); }
void ARM7opLDRBTofrmll() { ARM7addrLSofrmll(); ARM7opLDRBT(); }
void ARM7opLDRBTofrmlr() { ARM7addrLSofrmlr(); ARM7opLDRBT(); }
void ARM7opLDRBTofrmar() { ARM7addrLSofrmar(); ARM7opLDRBT(); }
void ARM7opLDRBTofrmrr() { ARM7addrLSofrmrr(); ARM7opLDRBT(); }
void ARM7opLDRBTofrpll() { ARM7addrLSofrpll(); ARM7opLDRBT(); }
void ARM7opLDRBTofrplr() { ARM7addrLSofrplr(); ARM7opLDRBT(); }
void ARM7opLDRBTofrpar() { ARM7addrLSofrpar(); ARM7opLDRBT(); }
void ARM7opLDRBTofrprr() { ARM7addrLSofrprr(); ARM7opLDRBT(); }

void ARM7opSTRBTptrmll() { ARM7addrLSptrmll(); ARM7opSTRBT(); }
void ARM7opSTRBTptrmlr() { ARM7addrLSptrmlr(); ARM7opSTRBT(); }
void ARM7opSTRBTptrmar() { ARM7addrLSptrmar(); ARM7opSTRBT(); }
void ARM7opSTRBTptrmrr() { ARM7addrLSptrmrr(); ARM7opSTRBT(); }
void ARM7opSTRBTptrpll() { ARM7addrLSptrpll(); ARM7opSTRBT(); }
void ARM7opSTRBTptrplr() { ARM7addrLSptrplr(); ARM7opSTRBT(); }
void ARM7opSTRBTptrpar() { ARM7addrLSptrpar(); ARM7opSTRBT(); }
void ARM7opSTRBTptrprr() { ARM7addrLSptrprr(); ARM7opSTRBT(); }
void ARM7opSTRBTprrmll() { ARM7addrLSprrmll(); ARM7opSTRBT(); }
void ARM7opSTRBTprrmlr() { ARM7addrLSprrmlr(); ARM7opSTRBT(); }
void ARM7opSTRBTprrmar() { ARM7addrLSprrmar(); ARM7opSTRBT(); }
void ARM7opSTRBTprrmrr() { ARM7addrLSprrmrr(); ARM7opSTRBT(); }
void ARM7opSTRBTprrpll() { ARM7addrLSprrpll(); ARM7opSTRBT(); }
void ARM7opSTRBTprrplr() { ARM7addrLSprrplr(); ARM7opSTRBT(); }
void ARM7opSTRBTprrpar() { ARM7addrLSprrpar(); ARM7opSTRBT(); }
void ARM7opSTRBTprrprr() { ARM7addrLSprrprr(); ARM7opSTRBT(); }
void ARM7opSTRBTofrmll() { ARM7addrLSofrmll(); ARM7opSTRBT(); }
void ARM7opSTRBTofrmlr() { ARM7addrLSofrmlr(); ARM7opSTRBT(); }
void ARM7opSTRBTofrmar() { ARM7addrLSofrmar(); ARM7opSTRBT(); }
void ARM7opSTRBTofrmrr() { ARM7addrLSofrmrr(); ARM7opSTRBT(); }
void ARM7opSTRBTofrpll() { ARM7addrLSofrpll(); ARM7opSTRBT(); }
void ARM7opSTRBTofrplr() { ARM7addrLSofrplr(); ARM7opSTRBT(); }
void ARM7opSTRBTofrpar() { ARM7addrLSofrpar(); ARM7opSTRBT(); }
void ARM7opSTRBTofrprr() { ARM7addrLSofrprr(); ARM7opSTRBT(); }

//-------------------------------------------------------------------------

void ARM7opMUL()
{
    arm7reg.r[ARM7OP_RN]=arm7reg.r[ARM7OP_RS]*arm7reg.r[ARM7OP_RM];
}

void ARM7opMLA()
{
    arm7reg.r[ARM7OP_RN]=arm7reg.r[ARM7OP_RS]*arm7reg.r[ARM7OP_RM]+arm7reg.r[ARM7OP_RD];
}

void ARM7opMULS()
{
    arm7reg.r[ARM7OP_RD]=arm7reg.r[ARM7OP_RS]*arm7reg.r[ARM7OP_RM];
    arm7reg.flags[ARMFLAG_N]=arm7reg.r[ARM7OP_RD]>>31;
    arm7reg.flags[ARMFLAG_Z]=arm7reg.r[ARM7OP_RD]?0:1;
}

void ARM7opMLAS()
{
    arm7reg.r[ARM7OP_RD]=arm7reg.r[ARM7OP_RS]*arm7reg.r[ARM7OP_RM]+arm7reg.r[ARM7OP_RN];
    arm7reg.flags[ARMFLAG_N]=arm7reg.r[ARM7OP_RD]>>31;
    arm7reg.flags[ARMFLAG_Z]=arm7reg.r[ARM7OP_RD]?0:1;
}

void ARM7opUMULL()
{
    __int64 t=arm7reg.r[ARM7OP_RS]*arm7reg.r[ARM7OP_RM];
    arm7reg.r[ARM7OP_RD]=t>>32;
    arm7reg.r[ARM7OP_RN]=t&0xFFFFFFFF;
}

void ARM7opUMLAL()
{
    __int64 t=arm7reg.r[ARM7OP_RS]*arm7reg.r[ARM7OP_RM];
    arm7reg.r[ARM7OP_RD]+=t>>32;
    arm7reg.r[ARM7OP_RN]+=t&0xFFFFFFFF;
}

void ARM7opSMULL()
{
    __int64 t=((signed)arm7reg.r[ARM7OP_RS])*((signed)arm7reg.r[ARM7OP_RM]);
    arm7reg.r[ARM7OP_RD]=t>>32;
    arm7reg.r[ARM7OP_RN]=t&0xFFFFFFFF;
}

void ARM7opSMLAL()
{
    __int64 t=((signed)arm7reg.r[ARM7OP_RS])*((signed)arm7reg.r[ARM7OP_RM]);
    arm7reg.r[ARM7OP_RD]+=t>>32;
    arm7reg.r[ARM7OP_RN]+=t&0xFFFFFFFF;
}

void ARM7opUMULLS()
{
    __int64 t=arm7reg.r[ARM7OP_RS]*arm7reg.r[ARM7OP_RM];
    arm7reg.r[ARM7OP_RD]=t>>32;
    arm7reg.r[ARM7OP_RN]=t&0xFFFFFFFF;
    arm7reg.flags[ARMFLAG_N]=arm7reg.r[ARM7OP_RD]>>31;
    arm7reg.flags[ARMFLAG_Z]=(arm7reg.r[ARM7OP_RD]|arm7reg.r[ARM7OP_RN])?0:1;
}

void ARM7opUMLALS()
{
    __int64 t=arm7reg.r[ARM7OP_RS]*arm7reg.r[ARM7OP_RM];
    arm7reg.r[ARM7OP_RD]+=t>>32;
    arm7reg.r[ARM7OP_RN]+=t&0xFFFFFFFF;
    arm7reg.flags[ARMFLAG_N]=arm7reg.r[ARM7OP_RD]>>31;
    arm7reg.flags[ARMFLAG_Z]=(arm7reg.r[ARM7OP_RD]|arm7reg.r[ARM7OP_RN])?0:1;
}

void ARM7opSMULLS()
{
    __int64 t=((signed)arm7reg.r[ARM7OP_RS])*((signed)arm7reg.r[ARM7OP_RM]);
    arm7reg.r[ARM7OP_RD]=t>>32;
    arm7reg.r[ARM7OP_RN]=t&0xFFFFFFFF;
    arm7reg.flags[ARMFLAG_N]=arm7reg.r[ARM7OP_RD]>>31;
    arm7reg.flags[ARMFLAG_Z]=(arm7reg.r[ARM7OP_RD]|arm7reg.r[ARM7OP_RN])?0:1;
}

void ARM7opSMLALS()
{
    __int64 t=((signed)arm7reg.r[ARM7OP_RS])*((signed)arm7reg.r[ARM7OP_RM]);
    arm7reg.r[ARM7OP_RD]+=t>>32;
    arm7reg.r[ARM7OP_RN]+=t&0xFFFFFFFF;
    arm7reg.flags[ARMFLAG_N]=arm7reg.r[ARM7OP_RD]>>31;
    arm7reg.flags[ARMFLAG_Z]=(arm7reg.r[ARM7OP_RD]|arm7reg.r[ARM7OP_RN])?0:1;
}

//-------------------------------------------------------------------------

int SideSum(u16 val)
{
    int res=val;
    u16 S[]={1,2,4,8},B[]={0x5555,0x3333,0x0F0F,0x00FF};
    res=((res>>S[0])&B[0])+(res&B[0]);
    res=((res>>S[1])&B[1])+(res&B[1]);
    res=((res>>S[2])&B[2])+(res&B[2]);
    res=((res>>S[3])&B[3])+(res&B[3]);
    return res;
}

#define ARM7addrLMIA() int a; arm7reg.tmp1=arm7reg.r[ARM7OP_RN]
#define ARM7addrLMIB() int a; arm7reg.tmp1=arm7reg.r[ARM7OP_RN]+4
#define ARM7addrLMDA() int a; arm7reg.tmp1=arm7reg.r[ARM7OP_RN]-(SideSum(arm7reg.curop&65535)*4)+4
#define ARM7addrLMDB() int a; arm7reg.tmp1=arm7reg.r[ARM7OP_RN]-(SideSum(arm7reg.curop&65535)*4)

#define ARM7addrLMwi() arm7reg.r[ARM7OP_RN]+=SideSum(arm7reg.curop&65535)*4
#define ARM7addrLMwd() arm7reg.r[ARM7OP_RN]-=SideSum(arm7reg.curop&65535)*4

#define ARM7opLDM() \
    for(a=0;a<=14;a++) \
    { \
        if(arm7reg.curop&(1<<a)) \
        { \
            arm7reg.r[a]=MMUrdW(0,arm7reg.tmp1); \
            arm7reg.tmp1+=4; \
        } \
    } \
    if(arm7reg.curop&0x00008000) \
    { \
        arm7reg.r[15]=MMUrdW(0,arm7reg.tmp1)&0xFFFFFFFC; \
        arm7reg.tmp1+=4; \
    }

#define ARM7opSTM() \
    for(a=0;a<=15;a++) \
    { \
        if(arm7reg.curop&(1<<a)) \
        { \
            MMUwrW(0,arm7reg.tmp1,arm7reg.r[a]); \
            arm7reg.tmp1+=4; \
        } \
    }

#define ARM7opLDMu() \
    if(arm7reg.curop&0x0001) { arm7reg.r[0]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0002) { arm7reg.r[1]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0004) { arm7reg.r[2]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0008) { arm7reg.r[3]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0010) { arm7reg.r[4]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0020) { arm7reg.r[5]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0040) { arm7reg.r[6]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0080) { arm7reg.r[7]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    switch(arm7reg.curmode) \
    { \
    	case ARMMD_USR: case ARMMD_SYS: \
    	    if(arm7reg.curop&0x0100) { arm7reg.r[8]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0200) { arm7reg.r[9]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0400) { arm7reg.r[10]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x0800) { arm7reg.r[11]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x1000) { arm7reg.r[12]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x2000) { arm7reg.r[13]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x4000) { arm7reg.r[14]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    break;\
    	case ARMMD_IRQ: case ARMMD_ABT: case ARMMD_SVC: case ARMMD_UND: \
    	    if(arm7reg.curop&0x0100) { arm7reg.r[8]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0200) { arm7reg.r[9]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0400) { arm7reg.r[10]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x0800) { arm7reg.r[11]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x1000) { arm7reg.r[12]=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x2000) { arm7reg.r13=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; }   \
    	    if(arm7reg.curop&0x4000) { arm7reg.r14=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; }   \
            break; \
    	case ARMMD_FIQ: \
    	    if(arm7reg.curop&0x0100) { arm7reg.r8=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0200) { arm7reg.r9=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0400) { arm7reg.r10=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x0800) { arm7reg.r11=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x1000) { arm7reg.r12=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x2000) { arm7reg.r13=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x4000) { arm7reg.r14=MMUrdW(0,arm7reg.tmp1); arm7reg.tmp1+=4; } \
            break; \
    } \
    if(arm7reg.curop&0x8000) \
    { \
    	arm7reg.r[15]=MMUrdW(0,arm7reg.tmp1)&0xFFFFFFFC; arm7reg.tmp1+=4; \
    	if(arm7reg.curmode>1) arm7reg.cpsr=arm7reg.spsr[arm7reg.curmode]; \
    }

#define ARM7opSTMu() \
    if(arm7reg.curop&0x0001) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[0]); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0002) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[1]); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0004) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[2]); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0008) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[3]); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0010) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[4]); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0020) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[5]); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0040) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[6]); arm7reg.tmp1+=4; } \
    if(arm7reg.curop&0x0080) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[7]); arm7reg.tmp1+=4; } \
    switch(arm7reg.curmode) \
    { \
    	case ARMMD_USR: case ARMMD_SYS: \
    	    if(arm7reg.curop&0x0100) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[8]); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0200) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[9]); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0400) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[10]); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x0800) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[11]); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x1000) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[12]); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x2000) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[13]); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x4000) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[14]); arm7reg.tmp1+=4; } \
    	    break; \
    	case ARMMD_IRQ: case ARMMD_ABT: case ARMMD_SVC: case ARMMD_UND: \
    	    if(arm7reg.curop&0x0100) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[8]); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0200) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[9]); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0400) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[10]); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x0800) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[11]); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x1000) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[12]); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x2000) { MMUwrW(0,arm7reg.tmp1,arm7reg.r13); arm7reg.tmp1+=4; }   \
    	    if(arm7reg.curop&0x4000) { MMUwrW(0,arm7reg.tmp1,arm7reg.r14); arm7reg.tmp1+=4; }   \
            break; \
    	case ARMMD_FIQ: \
    	    if(arm7reg.curop&0x0100) { MMUwrW(0,arm7reg.tmp1,arm7reg.r8); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0200) { MMUwrW(0,arm7reg.tmp1,arm7reg.r9); arm7reg.tmp1+=4;  } \
    	    if(arm7reg.curop&0x0400) { MMUwrW(0,arm7reg.tmp1,arm7reg.r10); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x0800) { MMUwrW(0,arm7reg.tmp1,arm7reg.r11); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x1000) { MMUwrW(0,arm7reg.tmp1,arm7reg.r12); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x2000) { MMUwrW(0,arm7reg.tmp1,arm7reg.r13); arm7reg.tmp1+=4; } \
    	    if(arm7reg.curop&0x4000) { MMUwrW(0,arm7reg.tmp1,arm7reg.r14); arm7reg.tmp1+=4; } \
            break; \
    } \
    if(arm7reg.curop&0x8000) { MMUwrW(0,arm7reg.tmp1,arm7reg.r[15]); arm7reg.tmp1+=4; }

void ARM7opLDMIA() { ARM7addrLMIA(); ARM7opLDM(); }
void ARM7opLDMIB() { ARM7addrLMIB(); ARM7opLDM(); }
void ARM7opLDMDA() { ARM7addrLMDA(); ARM7opLDM(); }
void ARM7opLDMDB() { ARM7addrLMDB(); ARM7opLDM(); }

void ARM7opLDMIAw() { ARM7addrLMIA(); ARM7addrLMwi(); ARM7opLDM(); }
void ARM7opLDMIBw() { ARM7addrLMIB(); ARM7addrLMwi(); ARM7opLDM(); }
void ARM7opLDMDAw() { ARM7addrLMDA(); ARM7addrLMwd(); ARM7opLDM(); }
void ARM7opLDMDBw() { ARM7addrLMDB(); ARM7addrLMwd(); ARM7opLDM(); }

void ARM7opLDMIAu() { ARM7addrLMIA(); ARM7opLDMu(); }
void ARM7opLDMIBu() { ARM7addrLMIB(); ARM7opLDMu(); }
void ARM7opLDMDAu() { ARM7addrLMDA(); ARM7opLDMu(); }
void ARM7opLDMDBu() { ARM7addrLMDB(); ARM7opLDMu(); }

void ARM7opLDMIAuw() { ARM7addrLMIA(); ARM7addrLMwi(); ARM7opLDMu(); }
void ARM7opLDMIBuw() { ARM7addrLMIB(); ARM7addrLMwi(); ARM7opLDMu(); }
void ARM7opLDMDAuw() { ARM7addrLMDA(); ARM7addrLMwd(); ARM7opLDMu(); }
void ARM7opLDMDBuw() { ARM7addrLMDB(); ARM7addrLMwd(); ARM7opLDMu(); }

void ARM7opSTMIA() { ARM7addrLMIA(); ARM7opSTM(); }
void ARM7opSTMIB() { ARM7addrLMIB(); ARM7opSTM(); }
void ARM7opSTMDA() { ARM7addrLMDA(); ARM7opSTM(); }
void ARM7opSTMDB() { ARM7addrLMDB(); ARM7opSTM(); }

void ARM7opSTMIAw() { ARM7addrLMIA(); ARM7addrLMwi(); ARM7opSTM(); }
void ARM7opSTMIBw() { ARM7addrLMIB(); ARM7addrLMwi(); ARM7opSTM(); }
void ARM7opSTMDAw() { ARM7addrLMDA(); ARM7addrLMwd(); ARM7opSTM(); }
void ARM7opSTMDBw() { ARM7addrLMDB(); ARM7addrLMwd(); ARM7opSTM(); }

void ARM7opSTMIAu() { ARM7addrLMIA(); ARM7opSTMu(); }
void ARM7opSTMIBu() { ARM7addrLMIB(); ARM7opSTMu(); }
void ARM7opSTMDAu() { ARM7addrLMDA(); ARM7opSTMu(); }
void ARM7opSTMDBu() { ARM7addrLMDB(); ARM7opSTMu(); }

void ARM7opSTMIAuw() { ARM7addrLMIA(); ARM7addrLMwi(); ARM7opSTMu(); }
void ARM7opSTMIBuw() { ARM7addrLMIB(); ARM7addrLMwi(); ARM7opSTMu(); }
void ARM7opSTMDAuw() { ARM7addrLMDA(); ARM7addrLMwd(); ARM7opSTMu(); }
void ARM7opSTMDBuw() { ARM7addrLMDB(); ARM7addrLMwd(); ARM7opSTMu(); }

//-------------------------------------------------------------------------

void ARM7opMRSrs()
{
    if(arm7reg.curmode>1) arm7reg.tmp1=arm7reg.spsr[arm7reg.curmode];
    else arm7reg.tmp1=arm7reg.cpsr;
    arm7reg.r[ARM7OP_RD]=arm7reg.cpsr;
}

void ARM7opMRSrc()
{
    arm7reg.r[ARM7OP_RD]=arm7reg.cpsr;
}

void ARM7opMSRic()
{
    arm7reg.tmp3 = arm7reg.curop&255;
    arm7reg.tmp4 = (arm7reg.curop&0x00000F00)>>7;
    arm7reg.tmp1 = (arm7reg.tmp3>>arm7reg.tmp4)|((arm7reg.tmp3&((1<<arm7reg.tmp4)-1))<<(32-arm7reg.tmp4));
    if(!(arm7reg.curop&0x00080000)) arm7reg.tmp1&=0x00FFFFFF;
    if(arm7reg.curmode)
    {
        if(!(arm7reg.curop&0x00040000)) arm7reg.tmp1&=0xFF00FFFF;
        if(!(arm7reg.curop&0x00020000)) arm7reg.tmp1&=0xFFFF00FF;
        if(!(arm7reg.curop&0x00010000)) arm7reg.tmp1&=0xFFFFFF00;
    }
    arm7reg.cpsr=arm7reg.tmp1;
    ARM7modesw(arm7reg.curmode,ARMcpsrtomode[arm7reg.cpsr&0x0F]);
}

void ARM7opMSRis()
{
    arm7reg.tmp3 = arm7reg.curop&255;
    arm7reg.tmp4 = (arm7reg.curop&0x00000F00)>>7;
    arm7reg.tmp1 = (arm7reg.tmp3>>arm7reg.tmp4)|((arm7reg.tmp3&((1<<arm7reg.tmp4)-1))<<(32-arm7reg.tmp4));
    if(!(arm7reg.curop&0x00080000)) arm7reg.tmp1&=0x00FFFFFF;
    if(arm7reg.curmode)
    {
        if(!(arm7reg.curop&0x00040000)) arm7reg.tmp1&=0xFF00FFFF;
        if(!(arm7reg.curop&0x00020000)) arm7reg.tmp1&=0xFFFF00FF;
        if(!(arm7reg.curop&0x00010000)) arm7reg.tmp1&=0xFFFFFF00;
    }
    if(arm7reg.curmode>1) arm7reg.spsr[arm7reg.curmode]=arm7reg.tmp1;
}

void ARM7opMSRrc()
{
    arm7reg.tmp1=arm7reg.r[ARM7OP_RM];
    if(!(arm7reg.curop&0x00080000)) arm7reg.tmp1&=0x00FFFFFF;
    if(arm7reg.curmode)
    {
        if(!(arm7reg.curop&0x00040000)) arm7reg.tmp1&=0xFF00FFFF;
        if(!(arm7reg.curop&0x00020000)) arm7reg.tmp1&=0xFFFF00FF;
        if(!(arm7reg.curop&0x00010000)) arm7reg.tmp1&=0xFFFFFF00;
    }
    arm7reg.cpsr=arm7reg.tmp1;
    ARM7modesw(arm7reg.curmode,ARMcpsrtomode[arm7reg.cpsr&0x0F]);
}

void ARM7opMSRrs()
{
    arm7reg.tmp1=arm7reg.r[ARM7OP_RM];
    if(!(arm7reg.curop&0x00080000)) arm7reg.tmp1&=0x00FFFFFF;
    if(arm7reg.curmode)
    {
        if(!(arm7reg.curop&0x00040000)) arm7reg.tmp1&=0xFF00FFFF;
        if(!(arm7reg.curop&0x00020000)) arm7reg.tmp1&=0xFFFF00FF;
        if(!(arm7reg.curop&0x00010000)) arm7reg.tmp1&=0xFFFFFF00;
    }
    if(arm7reg.curmode>1) arm7reg.spsr[arm7reg.curmode]=arm7reg.tmp1;
}

//-------------------------------------------------------------------------

#define ARM7addrSWP() \
    arm7reg.tmp2=arm7reg.r[ARM7OP_RN]; \
    switch(arm7reg.tmp2&3) \
    { \
        case 0: arm7reg.tmp1=arm7reg.tmp2; break; \
	case 1: arm7reg.tmp1=(arm7reg.tmp2>> 8)|((arm7reg.tmp2&0x000000FF)<<24); break; \
	case 2: arm7reg.tmp1=(arm7reg.tmp2>>16)|((arm7reg.tmp2&0x0000FFFF)<<16); break; \
	case 3: arm7reg.tmp1=(arm7reg.tmp2>>24)|((arm7reg.tmp2&0x00FFFFFF)<< 8); break; \
    }

void ARM7opSWP()
{
    ARM7addrSWP();
    arm7reg.tmp4=MMUrdW(0,arm7reg.tmp1);
    MMUwrW(0,arm7reg.tmp1,arm7reg.r[ARM7OP_RM]);
    arm7reg.r[ARM7OP_RD]=arm7reg.tmp4;
}

void ARM7opSWPB()
{
    ARM7addrSWP();
    arm7reg.tmp4=0|MMUrdB(0,arm7reg.tmp1);
    MMUwrB(0,arm7reg.tmp1,arm7reg.r[ARM7OP_RM]&255);
    arm7reg.r[ARM7OP_RD]=0|arm7reg.tmp4;
}

//-------------------------------------------------------------------------

void ARM7opSWI()
{
    arm7reg.r14svc=arm7reg.r[15];
    arm7reg.spsr[ARMMD_SVC]=arm7reg.cpsr;
    ARM7modesw(arm7reg.curmode, ARMMD_SVC);
    arm7reg.cpsr&=(0xFFFFFFFF-ARMS_T);
    arm7reg.cpsr|=ARMS_I;
    arm7reg.r[15]=0x00000008;
}

void ARM7irq()
{
    arm7reg.r14irq=arm7reg.r[15];
    arm7reg.spsr[ARMMD_IRQ]=arm7reg.cpsr;
    ARM7modesw(arm7reg.curmode, ARMMD_IRQ);
    arm7reg.cpsr&=(0xFFFFFFFF-ARMS_T);
    arm7reg.cpsr|=ARMS_I;
    arm7reg.r[15]=0x00000018;
}

//---Thumb time!-----------------------------------------------------------

void Thumb7opUND() {}

#define Thumb7addrImm3() \
    arm7reg.tmp1 = ARM7TOP_RM; \
    arm7reg.tmp2 = arm7reg.r[ARM7TOP_RN]

#define Thumb7addrImm5shft() \
    arm7reg.tmp1 = (arm7reg.curop&0x07C0)>>6; \
    if(!arm7reg.tmp1) arm7reg.tmp1=32; \
    arm7reg.tmp2 = arm7reg.r[ARM7TOP_RN]

#define Thumb7addrImm5(shft) \
    arm7reg.tmp1 = ((arm7reg.curop&0x07C0)>>4)>>(shft); \
    arm7reg.tmp2 = arm7reg.r[ARM7TOP_RN]

#define Thumb7addrImm7() \
    arm7reg.tmp1 = (arm7reg.curop&0x007F)<<2; \
    arm7reg.tmp2 = arm7reg.r[13]

#define Thumb7addrImm8(rg) \
    arm7reg.tmp1 = (arm7reg.curop&0x00FF); \
    arm7reg.tmp2 = arm7reg.r[(rg)]; \

#define Thumb7addrPC() \
    arm7reg.tmp1 = (arm7reg.curop&0x00FF)<<2; \
    arm7reg.tmp2 = (arm7reg.r[15]+2)&0xFFFFFFFC; \

#define Thumb7addrSP() \
    arm7reg.tmp1 = (arm7reg.curop&0x00FF)<<2; \
    arm7reg.tmp2 = arm7reg.r[13]

#define Thumb7addrReg() \
    arm7reg.tmp1 = arm7reg.r[ARM7TOP_RM]; \
    arm7reg.tmp2 = arm7reg.r[ARM7TOP_RN]

#define Thumb7addrH() \
    arm7reg.tmp1 = arm7reg.r[ARM7TOP_RNH]; \
    arm7reg.tmp2 = arm7reg.r[ARM7TOP_RDH]

#define Thumb7addrRegC() \
    arm7reg.tmp1=arm7reg.r[ARM7TOP_RN]; \
    arm7reg.tmp2=arm7reg.r[ARM7TOP_RD]; \
    arm7reg.tmpc=arm7reg.flags[ARMFLAG_C]

#define Thumb7addrB() \
    arm7reg.tmp2=arm7reg.curop&0x00FF; \
    arm7reg.tmp1=(arm7reg.tmp2&0x0080)? \
                 (0xFFFFFF00|arm7reg.tmp2): \
		 (0x00000000|arm7reg.tmp2)

#define Thumb7opAND() arm7reg.r[ARM7TOP_RD]=arm7reg.tmp1 & arm7reg.tmp2
#define Thumb7opEOR() arm7reg.r[ARM7TOP_RD]=arm7reg.tmp1 ^ arm7reg.tmp2
#define Thumb7opRSB() arm7reg.r[ARM7TOP_RD]=arm7reg.tmp1 - arm7reg.tmp2
#define Thumb7opADC() arm7reg.r[ARM7TOP_RD]=arm7reg.tmp1 + arm7reg.tmp2 + arm7reg.tmpc
#define Thumb7opSBC() arm7reg.r[ARM7TOP_RD]=arm7reg.tmp2 - arm7reg.tmp1 - arm7reg.tmpc
#define Thumb7opRSC() arm7reg.r[ARM7TOP_RD]=arm7reg.tmp1 - arm7reg.tmp2 + arm7reg.tmpc
#define Thumb7opTST() arm7reg.tmp3=arm7reg.tmp1 & arm7reg.tmp2
#define Thumb7opTEQ() arm7reg.tmp3=arm7reg.tmp1 ^ arm7reg.tmp2
#define Thumb7opCMP() arm7reg.tmp3=arm7reg.tmp2 - arm7reg.tmp1
#define Thumb7opCMN() arm7reg.tmp3=arm7reg.tmp1 + arm7reg.tmp2
#define Thumb7opORR() arm7reg.r[ARM7TOP_RD]=arm7reg.tmp1 | arm7reg.tmp2
#define Thumb7opBIC() arm7reg.r[ARM7TOP_RD]=arm7reg.tmp2 & ~(arm7reg.tmp1)
#define Thumb7opMVN() arm7reg.r[ARM7TOP_RD]=~(arm7reg.tmp1)
#define Thumb7opMUL() arm7reg.r[ARM7TOP_RD]=arm7reg.tmp1*arm7reg.tmp2
#define Thumb7opNEG() arm7reg.r[ARM7TOP_RD]=0-arm7reg.tmp1

#define Thumb7opADD(rg) arm7reg.r[(rg)]=arm7reg.tmp1 + arm7reg.tmp2
#define Thumb7opSUB(rg) arm7reg.r[(rg)]=arm7reg.tmp2 - arm7reg.tmp1
#define Thumb7opMOV(rg) arm7reg.r[(rg)]=arm7reg.tmp1

#define Thumb7opLDR(rg) arm7reg.r[(rg)]      =MMUrdW(0,arm7reg.tmp1+arm7reg.tmp2)
#define Thumb7opLDRB()  arm7reg.r[ARM7TOP_RD]=MMUrdB(0,arm7reg.tmp1+arm7reg.tmp2)
#define Thumb7opLDRH()  arm7reg.r[ARM7TOP_RD]=MMUrdH(0,arm7reg.tmp1+arm7reg.tmp2)

#define Thumb7opSTR(rg) MMUwrW(0,arm7reg.tmp1+arm7reg.tmp2,arm7reg.r[(rg)])
#define Thumb7opSTRB()  MMUwrB(0,arm7reg.tmp1+arm7reg.tmp2,arm7reg.r[ARM7TOP_RD])
#define Thumb7opSTRH()  MMUwrH(0,arm7reg.tmp1+arm7reg.tmp2,arm7reg.r[ARM7TOP_RD])

#define Thumb7opLDRSB() \
    arm7reg.r[ARM7TOP_RD]=MMUrdB(0,arm7reg.tmp1+arm7reg.tmp2); \
    if(arm7reg.r[ARM7TOP_RD]&0x80) arm7reg.r[ARM7TOP_RD]|=0xFFFFFF00

#define Thumb7opLDRSH() \
    arm7reg.r[ARM7TOP_RD]=MMUrdH(0,arm7reg.tmp1+arm7reg.tmp2); \
    if(arm7reg.r[ARM7TOP_RD]&0x8000) arm7reg.r[ARM7TOP_RD]|=0xFFFF0000

#define Thumb7opASR() \
    if(arm7reg.tmp1) { \
        if(arm7reg.tmp1<=32) { \
           arm7reg.r[ARM7TOP_RD]=(signed)arm7reg.tmp2>>arm7reg.tmp1; \
           arm7reg.tmpc=(arm7reg.tmp2&(1<<(arm7reg.tmp1-1)))?1:0; \
        } else { \
            arm7reg.tmpc=(arm7reg.tmp2&0x80000000)?1:0; \
            arm7reg.r[ARM7TOP_RD]=(arm7reg.tmpc)?0xFFFFFFFF:0; \
        } \
    }

#define Thumb7opLSL() \
    if(arm7reg.tmp1) { \
        if(arm7reg.tmp1<=32) { \
           arm7reg.r[ARM7TOP_RD]=arm7reg.tmp2<<arm7reg.tmp1; \
           arm7reg.tmpc=(arm7reg.tmp2&(1<<(32-arm7reg.tmp1)))?1:0; \
        } else { \
            arm7reg.tmpc=0; \
            arm7reg.r[ARM7TOP_RD]=0; \
        } \
    }

#define Thumb7opLSR() \
    if(arm7reg.tmp1) { \
        if(arm7reg.tmp1<=32) { \
           arm7reg.r[ARM7TOP_RD]=arm7reg.tmp2>>arm7reg.tmp1; \
           arm7reg.tmpc=(arm7reg.tmp2&(1<<(arm7reg.tmp1-1)))?1:0; \
        } else { \
            arm7reg.tmpc=0; \
            arm7reg.r[ARM7TOP_RD]=0; \
        } \
    }

#define Thumb7opROR() \
    if(arm7reg.tmp1&255) { \
        arm7reg.tmp1&=31; \
        if(arm7reg.tmp1) { \
           arm7reg.r[ARM7TOP_RD]=(arm7reg.tmp2>>arm7reg.tmp1)|((arm7reg.tmp2&((1<<arm7reg.tmp1)-1))<<(32-arm7reg.tmp1)); \
           arm7reg.tmpc = (arm7reg.tmp2>>(arm7reg.tmp1-1))&1; \
        } else { \
            arm7reg.tmpc=(arm7reg.tmp2&0x80000000)?1:0; \
            arm7reg.r[ARM7TOP_RD]=0; \
        } \
    }

#define Thumb7opLDMIA(rg) \
    int a; \
    arm7reg.tmp1=arm7reg.r[(rg)]; \
    arm7reg.r[(rg)]+=SideSum(arm7reg.curop&0x00FF)*4; \
    for(a=0;a<=7;a++) \
    { \
        if(arm7reg.curop&(1<<a)) \
        { \
            arm7reg.r[a]=MMUrdW(0,arm7reg.tmp1); \
            arm7reg.tmp1+=4; \
        } \
    } \

#define Thumb7opSTMIA(rg) \
    int a; \
    arm7reg.tmp1=arm7reg.r[(rg)]; \
    arm7reg.r[(rg)]+=SideSum(arm7reg.curop&0x00FF)*4; \
    for(a=0;a<=7;a++) \
    { \
        if(arm7reg.curop&(1<<a)) \
        { \
            MMUwrW(0,arm7reg.tmp1,arm7reg.r[a]); \
            arm7reg.tmp1+=4; \
        } \
    } \

#define Thumb7flagsADC() \
    arm7reg.flags[ARMFLAG_N]=arm7reg.r[ARM7TOP_RD]>>31; \
    arm7reg.flags[ARMFLAG_Z]=arm7reg.r[ARM7TOP_RD]?0:1; \
    arm7reg.tmp2+=arm7reg.tmpc; \
    arm7reg.flags[ARMFLAG_V]=((arm7reg.tmp2^arm7reg.r[ARM7TOP_RD])&(arm7reg.tmp1^arm7reg.r[ARM7TOP_RD]))>>31; \
    arm7reg.flags[ARMFLAG_C]=((arm7reg.tmp2>>31)+(arm7reg.tmp1>>31)+arm7reg.flags[ARMFLAG_V])>>1

#define Thumb7flagsADD(rg) \
    arm7reg.flags[ARMFLAG_N]=arm7reg.r[(rg)]>>31; \
    arm7reg.flags[ARMFLAG_Z]=arm7reg.r[(rg)]?0:1; \
    arm7reg.flags[ARMFLAG_V]=((arm7reg.tmp2^arm7reg.r[(rg)])&(arm7reg.tmp1^arm7reg.r[(rg)]))>>31; \
    arm7reg.flags[ARMFLAG_C]=((arm7reg.tmp2>>31)+(arm7reg.tmp1>>31)+arm7reg.flags[ARMFLAG_V])>>1

#define Thumb7flagsMOV(rg) \
    arm7reg.flags[ARMFLAG_N]=arm7reg.r[(rg)]>>31; \
    arm7reg.flags[ARMFLAG_Z]=arm7reg.r[(rg)]?0:1

#define Thumb7flagsBIT() \
    arm7reg.flags[ARMFLAG_N]=arm7reg.r[ARM7TOP_RD]>>31; \
    arm7reg.flags[ARMFLAG_Z]=arm7reg.r[ARM7TOP_RD]?0:1; \
    arm7reg.flags[ARMFLAG_C]=arm7reg.tmpc

#define Thumb7flagsSUB(rg) \
    arm7reg.flags[ARMFLAG_N]=arm7reg.r[(rg)]>>31; \
    arm7reg.flags[ARMFLAG_Z]=arm7reg.r[(rg)]?0:1; \
    arm7reg.flags[ARMFLAG_C]=(arm7reg.tmp2<arm7reg.tmp1)?0:1; \
    arm7reg.flags[ARMFLAG_V]=((arm7reg.tmp2^arm7reg.tmp1)&(arm7reg.tmp2^arm7reg.r[(rg)]))>>31

#define Thumb7flagsSBC() \
    arm7reg.flags[ARMFLAG_N]=arm7reg.r[ARM7OP_RD]>>31; \
    arm7reg.flags[ARMFLAG_Z]=arm7reg.r[ARM7OP_RD]?0:1; \
    arm7reg.tmp1+=arm7reg.tmpc; \
    arm7reg.flags[ARMFLAG_V]=((arm7reg.tmp2^arm7reg.tmp1)&(arm7reg.tmp2^arm7reg.r[ARM7OP_RD]))>>31; \
    arm7reg.flags[ARMFLAG_C]=((arm7reg.tmp2>>31)+(arm7reg.tmp1>>31)+arm7reg.flags[ARMFLAG_V])>>1

#define Thumb7flagsCMN() \
    arm7reg.flags[ARMFLAG_N]=arm7reg.tmp3>>31; \
    arm7reg.flags[ARMFLAG_Z]=arm7reg.tmp3?0:1; \
    arm7reg.flags[ARMFLAG_V]=((arm7reg.tmp2^arm7reg.tmp3)&(arm7reg.tmp1^arm7reg.tmp3))>>31; \
    arm7reg.flags[ARMFLAG_C]=((arm7reg.tmp2>>31)+(arm7reg.tmp1>>31)+arm7reg.flags[ARMFLAG_V])>>1

#define Thumb7flagsCMP() \
    arm7reg.flags[ARMFLAG_N]=arm7reg.tmp3>>31; \
    arm7reg.flags[ARMFLAG_Z]=arm7reg.tmp3?0:1; \
    arm7reg.flags[ARMFLAG_C]=(arm7reg.tmp2<arm7reg.tmp1)?0:1; \
    arm7reg.flags[ARMFLAG_V]=((arm7reg.tmp2^arm7reg.tmp1)&(arm7reg.tmp2^arm7reg.tmp3))>>31

#define Thumb7flagsTST() \
    arm7reg.flags[ARMFLAG_N]=arm7reg.tmp3>>31; \
    arm7reg.flags[ARMFLAG_Z]=arm7reg.tmp3?0:1; \
    arm7reg.flags[ARMFLAG_C]=arm7reg.tmpc

void Thumb7opLSLimm() { Thumb7addrImm5shft(); Thumb7opLSL(); Thumb7flagsBIT(); }
void Thumb7opLSRimm() { Thumb7addrImm5shft(); Thumb7opLSR(); Thumb7flagsBIT(); }
void Thumb7opASRimm() { Thumb7addrImm5shft(); Thumb7opASR(); Thumb7flagsBIT(); }
void Thumb7opADDreg() { Thumb7addrReg(); Thumb7opADD(ARM7TOP_RD); Thumb7flagsADD(ARM7TOP_RD); }
void Thumb7opSUBreg() { Thumb7addrReg(); Thumb7opSUB(ARM7TOP_RD); Thumb7flagsSUB(ARM7TOP_RD); }

void Thumb7opADDimm3() { Thumb7addrImm3(); Thumb7opADD(ARM7TOP_RD); Thumb7flagsADD(ARM7TOP_RD); }
void Thumb7opSUBimm3() { Thumb7addrImm3(); Thumb7opSUB(ARM7TOP_RD); Thumb7flagsSUB(ARM7TOP_RD); }

void Thumb7opMOVimm8r0() { Thumb7addrImm8(0); Thumb7opMOV(0); Thumb7flagsMOV(0); }
void Thumb7opMOVimm8r1() { Thumb7addrImm8(1); Thumb7opMOV(1); Thumb7flagsMOV(1); }
void Thumb7opMOVimm8r2() { Thumb7addrImm8(2); Thumb7opMOV(2); Thumb7flagsMOV(2); }
void Thumb7opMOVimm8r3() { Thumb7addrImm8(3); Thumb7opMOV(3); Thumb7flagsMOV(3); }
void Thumb7opMOVimm8r4() { Thumb7addrImm8(4); Thumb7opMOV(4); Thumb7flagsMOV(4); }
void Thumb7opMOVimm8r5() { Thumb7addrImm8(5); Thumb7opMOV(5); Thumb7flagsMOV(5); }
void Thumb7opMOVimm8r6() { Thumb7addrImm8(6); Thumb7opMOV(6); Thumb7flagsMOV(6); }
void Thumb7opMOVimm8r7() { Thumb7addrImm8(7); Thumb7opMOV(7); Thumb7flagsMOV(7); }
void Thumb7opCMPimm8r0() { Thumb7addrImm8(0); Thumb7opCMP();  Thumb7flagsCMP();  }
void Thumb7opCMPimm8r1() { Thumb7addrImm8(1); Thumb7opCMP();  Thumb7flagsCMP();  }
void Thumb7opCMPimm8r2() { Thumb7addrImm8(2); Thumb7opCMP();  Thumb7flagsCMP();  }
void Thumb7opCMPimm8r3() { Thumb7addrImm8(3); Thumb7opCMP();  Thumb7flagsCMP();  }
void Thumb7opCMPimm8r4() { Thumb7addrImm8(4); Thumb7opCMP();  Thumb7flagsCMP();  }
void Thumb7opCMPimm8r5() { Thumb7addrImm8(5); Thumb7opCMP();  Thumb7flagsCMP();  }
void Thumb7opCMPimm8r6() { Thumb7addrImm8(6); Thumb7opCMP();  Thumb7flagsCMP();  }
void Thumb7opCMPimm8r7() { Thumb7addrImm8(7); Thumb7opCMP();  Thumb7flagsCMP();  }
void Thumb7opADDimm8r0() { Thumb7addrImm8(0); Thumb7opADD(0); Thumb7flagsADD(0); }
void Thumb7opADDimm8r1() { Thumb7addrImm8(1); Thumb7opADD(1); Thumb7flagsADD(1); }
void Thumb7opADDimm8r2() { Thumb7addrImm8(2); Thumb7opADD(2); Thumb7flagsADD(2); }
void Thumb7opADDimm8r3() { Thumb7addrImm8(3); Thumb7opADD(3); Thumb7flagsADD(3); }
void Thumb7opADDimm8r4() { Thumb7addrImm8(4); Thumb7opADD(4); Thumb7flagsADD(4); }
void Thumb7opADDimm8r5() { Thumb7addrImm8(5); Thumb7opADD(5); Thumb7flagsADD(5); }
void Thumb7opADDimm8r6() { Thumb7addrImm8(6); Thumb7opADD(6); Thumb7flagsADD(6); }
void Thumb7opADDimm8r7() { Thumb7addrImm8(7); Thumb7opADD(7); Thumb7flagsADD(7); }
void Thumb7opSUBimm8r0() { Thumb7addrImm8(0); Thumb7opSUB(0); Thumb7flagsSUB(0); }
void Thumb7opSUBimm8r1() { Thumb7addrImm8(1); Thumb7opSUB(1); Thumb7flagsSUB(1); }
void Thumb7opSUBimm8r2() { Thumb7addrImm8(2); Thumb7opSUB(2); Thumb7flagsSUB(2); }
void Thumb7opSUBimm8r3() { Thumb7addrImm8(3); Thumb7opSUB(3); Thumb7flagsSUB(3); }
void Thumb7opSUBimm8r4() { Thumb7addrImm8(4); Thumb7opSUB(4); Thumb7flagsSUB(4); }
void Thumb7opSUBimm8r5() { Thumb7addrImm8(5); Thumb7opSUB(5); Thumb7flagsSUB(5); }
void Thumb7opSUBimm8r6() { Thumb7addrImm8(6); Thumb7opSUB(6); Thumb7flagsSUB(6); }
void Thumb7opSUBimm8r7() { Thumb7addrImm8(7); Thumb7opSUB(7); Thumb7flagsSUB(7); }

void Thumb7opDPg1()
{
    Thumb7addrRegC();
    switch((arm7reg.curop&0x00C0)>>6)
    {
        case 0: Thumb7opAND(); Thumb7flagsBIT(); break;
        case 1: Thumb7opEOR(); Thumb7flagsBIT(); break;
        case 2: Thumb7opLSL(); Thumb7flagsBIT(); break;
        case 3: Thumb7opLSR(); Thumb7flagsBIT(); break;
    }
}

void Thumb7opDPg2()
{
    Thumb7addrRegC();
    switch((arm7reg.curop&0x00C0)>>6)
    {
        case 0: Thumb7opASR(); Thumb7flagsBIT(); break;
        case 1: Thumb7opADC(); Thumb7flagsADC(); break;
        case 2: Thumb7opSBC(); Thumb7flagsSBC(); break;
        case 3: Thumb7opROR(); Thumb7flagsBIT(); break;
    }
}

void Thumb7opDPg3()
{
    Thumb7addrRegC();
    switch((arm7reg.curop&0x00C0)>>6)
    {
        case 0: Thumb7opTST(); Thumb7flagsBIT(); break;
        case 1: Thumb7opNEG(); Thumb7flagsSUB(ARM7TOP_RD); break;
        case 2: Thumb7opCMP(); Thumb7flagsCMP(); break;
        case 3: Thumb7opCMN(); Thumb7flagsCMN(); break;
    }
}

void Thumb7opDPg4()
{
    Thumb7addrRegC();
    switch((arm7reg.curop&0x00C0)>>6)
    {
        case 0: Thumb7opORR(); Thumb7flagsBIT(); break;
        case 1: Thumb7opMUL(); Thumb7flagsBIT(); break;
        case 2: Thumb7opBIC(); Thumb7flagsBIT(); break;
        case 3: Thumb7opMVN(); Thumb7flagsBIT(); break;
    }
}

void Thumb7opADDH() { Thumb7addrH(); Thumb7opADD(ARM7TOP_RDH); Thumb7flagsADD(ARM7TOP_RDH); }
void Thumb7opMOVH() { Thumb7addrH(); Thumb7opMOV(ARM7TOP_RDH); }
void Thumb7opCMPH() { Thumb7addrH(); Thumb7opCMP(); Thumb7flagsCMP(); }

void Thumb7opLDRPCr0() { Thumb7addrPC(); Thumb7opLDR(0); }
void Thumb7opLDRPCr1() { Thumb7addrPC(); Thumb7opLDR(1); }
void Thumb7opLDRPCr2() { Thumb7addrPC(); Thumb7opLDR(2); }
void Thumb7opLDRPCr3() { Thumb7addrPC(); Thumb7opLDR(3); }
void Thumb7opLDRPCr4() { Thumb7addrPC(); Thumb7opLDR(4); }
void Thumb7opLDRPCr5() { Thumb7addrPC(); Thumb7opLDR(5); }
void Thumb7opLDRPCr6() { Thumb7addrPC(); Thumb7opLDR(6); }
void Thumb7opLDRPCr7() { Thumb7addrPC(); Thumb7opLDR(7); }

void Thumb7opLDRSPr0() { Thumb7addrSP(); Thumb7opLDR(0); }
void Thumb7opLDRSPr1() { Thumb7addrSP(); Thumb7opLDR(1); }
void Thumb7opLDRSPr2() { Thumb7addrSP(); Thumb7opLDR(2); }
void Thumb7opLDRSPr3() { Thumb7addrSP(); Thumb7opLDR(3); }
void Thumb7opLDRSPr4() { Thumb7addrSP(); Thumb7opLDR(4); }
void Thumb7opLDRSPr5() { Thumb7addrSP(); Thumb7opLDR(5); }
void Thumb7opLDRSPr6() { Thumb7addrSP(); Thumb7opLDR(6); }
void Thumb7opLDRSPr7() { Thumb7addrSP(); Thumb7opLDR(7); }
void Thumb7opSTRSPr0() { Thumb7addrSP(); Thumb7opSTR(0); }
void Thumb7opSTRSPr1() { Thumb7addrSP(); Thumb7opSTR(1); }
void Thumb7opSTRSPr2() { Thumb7addrSP(); Thumb7opSTR(2); }
void Thumb7opSTRSPr3() { Thumb7addrSP(); Thumb7opSTR(3); }
void Thumb7opSTRSPr4() { Thumb7addrSP(); Thumb7opSTR(4); }
void Thumb7opSTRSPr5() { Thumb7addrSP(); Thumb7opSTR(5); }
void Thumb7opSTRSPr6() { Thumb7addrSP(); Thumb7opSTR(6); }
void Thumb7opSTRSPr7() { Thumb7addrSP(); Thumb7opSTR(7); }

void Thumb7opSTRreg()   { Thumb7addrReg(); Thumb7opSTR(ARM7TOP_RD); }
void Thumb7opLDRreg()   { Thumb7addrReg(); Thumb7opLDR(ARM7TOP_RD); }
void Thumb7opSTRHreg()  { Thumb7addrReg(); Thumb7opSTRH();  }
void Thumb7opLDRHreg()  { Thumb7addrReg(); Thumb7opLDRH();  }
void Thumb7opSTRBreg()  { Thumb7addrReg(); Thumb7opSTRB();  }
void Thumb7opLDRBreg()  { Thumb7addrReg(); Thumb7opLDRB();  }
void Thumb7opLDRSBreg() { Thumb7addrReg(); Thumb7opLDRSB(); }
void Thumb7opLDRSHreg() { Thumb7addrReg(); Thumb7opLDRSH(); }

void Thumb7opSTRimm5()  { Thumb7addrImm5(0); Thumb7opSTR(ARM7TOP_RD); }
void Thumb7opLDRimm5()  { Thumb7addrImm5(0); Thumb7opLDR(ARM7TOP_RD); }
void Thumb7opSTRHimm5() { Thumb7addrImm5(1); Thumb7opSTRH(); }
void Thumb7opLDRHimm5() { Thumb7addrImm5(1); Thumb7opLDRH(); }
void Thumb7opSTRBimm5() { Thumb7addrImm5(2); Thumb7opSTRB(); }
void Thumb7opLDRBimm5() { Thumb7addrImm5(2); Thumb7opLDRB(); }

void Thumb7opADDPCr0() { Thumb7addrPC(); Thumb7opADD(0); Thumb7flagsADD(0); }
void Thumb7opADDPCr1() { Thumb7addrPC(); Thumb7opADD(1); Thumb7flagsADD(1); }
void Thumb7opADDPCr2() { Thumb7addrPC(); Thumb7opADD(2); Thumb7flagsADD(2); }
void Thumb7opADDPCr3() { Thumb7addrPC(); Thumb7opADD(3); Thumb7flagsADD(3); }
void Thumb7opADDPCr4() { Thumb7addrPC(); Thumb7opADD(4); Thumb7flagsADD(4); }
void Thumb7opADDPCr5() { Thumb7addrPC(); Thumb7opADD(5); Thumb7flagsADD(5); }
void Thumb7opADDPCr6() { Thumb7addrPC(); Thumb7opADD(6); Thumb7flagsADD(6); }
void Thumb7opADDPCr7() { Thumb7addrPC(); Thumb7opADD(7); Thumb7flagsADD(7); }
void Thumb7opADDSPr0() { Thumb7addrSP(); Thumb7opADD(0); Thumb7flagsADD(0); }
void Thumb7opADDSPr1() { Thumb7addrSP(); Thumb7opADD(1); Thumb7flagsADD(1); }
void Thumb7opADDSPr2() { Thumb7addrSP(); Thumb7opADD(2); Thumb7flagsADD(2); }
void Thumb7opADDSPr3() { Thumb7addrSP(); Thumb7opADD(3); Thumb7flagsADD(3); }
void Thumb7opADDSPr4() { Thumb7addrSP(); Thumb7opADD(4); Thumb7flagsADD(4); }
void Thumb7opADDSPr5() { Thumb7addrSP(); Thumb7opADD(5); Thumb7flagsADD(5); }
void Thumb7opADDSPr6() { Thumb7addrSP(); Thumb7opADD(6); Thumb7flagsADD(6); }
void Thumb7opADDSPr7() { Thumb7addrSP(); Thumb7opADD(7); Thumb7flagsADD(7); }

void Thumb7opADDSPimm7() { Thumb7addrImm7(); Thumb7opADD(13); Thumb7flagsADD(13); }

void Thumb7opPUSH()
{
    int a;
    arm7reg.tmp2=0;
    arm7reg.tmp1=arm7reg.r[13]-SideSum(arm7reg.curop&0x00FF)*4;
    arm7reg.r[13]-=SideSum(arm7reg.curop&0x00ff)*4;
    for(a=0;a<=7;a++)
    {
        if(arm7reg.curop&(1<<a))
        {
            MMUwrW(0,arm7reg.tmp1,arm7reg.r[a]);
            arm7reg.tmp1+=4;
        }
    }
}

void Thumb7opPUSHlr()
{
    int a;
    arm7reg.tmp1=arm7reg.r[13]-SideSum(arm7reg.curop&0x01FF)*4;
    arm7reg.r[13]-=SideSum(arm7reg.curop&0x01ff)*4;
    for(a=0;a<=7;a++)
    {
        if(arm7reg.curop&(1<<a))
        {
            MMUwrW(0,arm7reg.tmp1,arm7reg.r[a]);
            arm7reg.tmp1+=4;
        }
    }
    if(arm7reg.curop&0x0100)
        MMUwrW(0,arm7reg.tmp1,arm7reg.r[14]);
}

void Thumb7opPOP()
{
    int a;
    arm7reg.tmp1=arm7reg.r[13];
    arm7reg.r[13]+=SideSum(arm7reg.curop&0x00FF)*4;
    for(a=0;a<=7;a++)
    {
        if(arm7reg.curop&(1<<a))
        {
            arm7reg.r[a]=MMUrdW(0,arm7reg.tmp1);
            arm7reg.tmp1+=4;
        }
    }
}

void Thumb7opPOPpc()
{
    int a;
    arm7reg.tmp1=arm7reg.r[13];
    arm7reg.r[13]+=SideSum(arm7reg.curop&0x01FF)*4;
    for(a=0;a<=7;a++)
    {
        if(arm7reg.curop&(1<<a))
        {
            arm7reg.r[a]=MMUrdW(0,arm7reg.tmp1);
            arm7reg.tmp1+=4;
        }
    }
    if(arm7reg.curop&0x0100)
    {
        arm7reg.r[15]=MMUrdW(0,arm7reg.tmp1)&0xFFFFFFFE;
        //ARM5: ThumbMode=r15&1
    }
}

void Thumb7opBKPT() {}
void Thumb7opSWI() {}

void Thumb7opLDMIAr0() {Thumb7opLDMIA(0);}
void Thumb7opLDMIAr1() {Thumb7opLDMIA(1);}
void Thumb7opLDMIAr2() {Thumb7opLDMIA(2);}
void Thumb7opLDMIAr3() {Thumb7opLDMIA(3);}
void Thumb7opLDMIAr4() {Thumb7opLDMIA(4);}
void Thumb7opLDMIAr5() {Thumb7opLDMIA(5);}
void Thumb7opLDMIAr6() {Thumb7opLDMIA(6);}
void Thumb7opLDMIAr7() {Thumb7opLDMIA(7);}
void Thumb7opSTMIAr0() {Thumb7opSTMIA(0);}
void Thumb7opSTMIAr1() {Thumb7opSTMIA(1);}
void Thumb7opSTMIAr2() {Thumb7opSTMIA(2);}
void Thumb7opSTMIAr3() {Thumb7opSTMIA(3);}
void Thumb7opSTMIAr4() {Thumb7opSTMIA(4);}
void Thumb7opSTMIAr5() {Thumb7opSTMIA(5);}
void Thumb7opSTMIAr6() {Thumb7opSTMIA(6);}
void Thumb7opSTMIAr7() {Thumb7opSTMIA(7);}

void Thumb7opBEQ() {if(ARM7condEQ()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;}}
void Thumb7opBNE() {if(ARM7condNE()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;}}
void Thumb7opBCS() {if(ARM7condCS()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;}}
void Thumb7opBCC() {if(ARM7condCC()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;}}
void Thumb7opBMI() {if(ARM7condMI()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;}}
void Thumb7opBPL() {if(ARM7condPL()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;}}
void Thumb7opBVS() {if(ARM7condVS()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;}}
void Thumb7opBVC() {if(ARM7condVC()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;}}
void Thumb7opBHI() {if(ARM7condHI()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;}}
void Thumb7opBLS() {if(ARM7condLS()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;}}
void Thumb7opBGE() {if(ARM7condGE()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;}}
void Thumb7opBLT() {if(ARM7condLT()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;}}
void Thumb7opBGT() {if(ARM7condGT()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;}}
void Thumb7opBLE() {if(ARM7condLE()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;}}
void Thumb7opBAL() {if(ARM7condAL()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;}}
void Thumb7opBNV() {if(ARM7condNV()){Thumb7addrB();arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;}}

void Thumb7opB()
{
    arm7reg.tmp2=arm7reg.curop&0x07FF;
    arm7reg.tmp1=(arm7reg.tmp2&0x0400)?
                 (0xFFFFF800|arm7reg.tmp2):
		 (0x00000000|arm7reg.tmp2);
    arm7reg.r[15]+=((signed)arm7reg.tmp1)*2+2;
}

void Thumb7opBXreg()
{
    if(arm7reg.curop&0x0080) arm7reg.r[14]=arm7reg.r[15]|1;
    arm7reg.r[15]=arm7reg.r[ARM7TOP_RNH]&0xFFFFFFFE;
    if(arm7reg.r[ARM7TOP_RNH]&1) arm7reg.cpsr|=ARMS_T;
                          else arm7reg.cpsr&=(0xFFFFFFFF-ARMS_T);
}

void Thumb7opBLsetup()
{
    arm7reg.tmp2=arm7reg.curop&0x07FF;
    arm7reg.tmp1=(arm7reg.tmp2&0x0400)?
                 (0xFFFFF800|arm7reg.tmp2):
		 (0x00000000|arm7reg.tmp2);
    arm7reg.r[14]=arm7reg.r[15]+((signed)arm7reg.tmp1)*4096+2;
}

void Thumb7opBLoff()
{
    arm7reg.tmp2=arm7reg.curop&0x07FF;
    arm7reg.tmp1=arm7reg.r[14];
    arm7reg.r[14]=arm7reg.r[15]|1;
    arm7reg.r[15]=arm7reg.tmp1+arm7reg.tmp2*2;
}

/*** EOF:arm7.c **********************************************************/

