/**************************************************************************
* DSemu: Memory management and handling (mmu.c)                           *
* Released under the terms of the BSD Public Licence                      *
* Imran Nazar (tf@oopsilon.com), 2004                                     *
**************************************************************************/

#include <stdio.h>
#include "vtbl.h"
#include "err.h"
#include "mmu.h"
#include "dma.h"
#include "dsioreg.h"
#include "gpu.h"
#include "unzip.h"

//#define MMUDEBUG
//#define BIOSsize 1024

int mmuioimpl[]={
    1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
};

extern u16 *VRAM;
extern u8  *VRAM8;

extern u16 OAM[512];
extern u32 LCDcolour[32768];
extern u16 GPUPAL[1024];
extern u8 *GPUPAL8;

RAMWORD mmuEmpty;
u32 ROMsize; char *ROMfile;
u32 BIOSsize;

RAMWORD *MMUbanks[16]={NULL};
u32 MMUmask[16]={0x1FF,0,0x3FFFF,0x7FFF,0,0,0,0,
                 0xFFFFFF,0xFFFFFF,0xFFFFFF,
                 0xFFFFFF,0xFFFFFF,0xFFFFFF,0,0};

u32 VRAMmap[553];

RAMWORD DTCM[4096], ITCM[4096];

int MMUinit(char *file)
{
    FILE *fp; u8 c; int a;

    ROMfile=file;

    EWRAM=(RAMWORD*)malloc(1048576*sizeof(RAMWORD));
    if(!EWRAM) RETFAIL("FAIL: MMU: EWRAM allocation.");

    IWRAM=(RAMWORD*)malloc(8192*sizeof(RAMWORD));
    if(!IWRAM) RETFAIL("FAIL: MMU: IWRAM allocation.");

//    if(MMUreset()) RETFAIL("FAIL: MMU: Reset.");

    MMUbanks[0x0]=BIOS;    MMUbanks[0x2]=EWRAM;
    MMUbanks[0x3]=IWRAM;
    MMUbanks[0x8]=GameROM; MMUbanks[0x9]=GameROM;
    MMUbanks[0xA]=GameROM; MMUbanks[0xB]=GameROM;
    MMUbanks[0xC]=GameROM; MMUbanks[0xD]=GameROM;

    mmuEmpty.data=0xFFFFFFFF;
    mmuEmpty.op=ARM7opUNL;
    mmuEmpty.cond=ARM7condAL;
//    mmuEmpty.cyc=1;

    RETPASS("MMU: Memoryspace allocation complete.");
}

int MMUreset()
{
    HANDLE hFile; u8 c; int a,done=0,bytesread; u8 *tempROM;
    char BIOSpath[1024], INIpath[1024], str[256], strout[384]; int pathsize;
    HZIP hz; ZIPENTRY ze; int cnt;

    memset(EWRAM,0,0x3FFFFF);
    memset(IWRAM,0,0x7FFF);
    memset(DTCM,0,4096*sizeof(RAMWORD));

    pathsize=strstr(logvt->file,"\\log.txt")-logvt->file;
    strncpy(BIOSpath, logvt->file, pathsize); BIOSpath[pathsize]=0;
    sprintf(INIpath, "%s\\dsemu.ini", BIOSpath);
    GetPrivateProfileString("General","BIOS","bioshack.bin",str,256,INIpath);
    sprintf(BIOSpath, "%s\\%s", BIOSpath,str);

//    fp=fopen(ROMfile,"rb");
    if(strstr(ROMfile,".zip"))
    {
        hz=OpenZipFromName(ROMfile,0);
        if(!hz) RETFAIL("FAIL: MMU: Zip file open.");
        #ifdef MMUDEBUG
        logvt->append("MMU: Zip file open.");
        #endif
        GetZipItem(hz,-1,&ze); cnt=ze.index; a=0;
        do {
            GetZipItem(hz,a,&ze);
            if(ze.unc_size && (strstr(ze.name,".gba") || strstr(ze.name,".bin")))
	        done=1;
	    else a++;
	} while(!done && a<cnt);
	if(a==cnt) RETFAIL("FAIL: MMU: Zip file has no GBA binary.");

        ROMsize=ze.unc_size;
        sprintf(strout,"MMU: Zipped ROM: %s, %d bytes.",ze.name,ROMsize);
        logvt->append(strout);
        GameROM=(RAMWORD*)malloc((ROMsize/4)*sizeof(RAMWORD));
        tempROM=(u8*)malloc(ROMsize);
        if(!GameROM || !tempROM) RETFAIL("FAIL: MMU: GameROM allocation.");
        #ifdef MMUDEBUG
        logvt->append("MMU: GameROM allocated.");
        #endif

        a=UnzipItemToBlock(hz,a,tempROM,ROMsize); if(a && a!=ZR_MORE)
        {
	    FormatZipMessage(a,str,256);
	    sprintf(strout,"FAIL: MMU: Zip read: %s.",str);
	    RETFAIL(strout);
	}

        CloseZip(hz);
        #ifdef MMUDEBUG
        logvt->append("MMU: Zip read and closed.");
        #endif
    } else {
        hFile=CreateFile(ROMfile,GENERIC_READ,0,NULL,OPEN_EXISTING,0,NULL);
        if(hFile==INVALID_HANDLE_VALUE) {
            FormatMessage(
                          FORMAT_MESSAGE_FROM_SYSTEM |
                          FORMAT_MESSAGE_IGNORE_INSERTS,
                          NULL, GetLastError(), 0, (LPTSTR)str, 256, NULL);
            sprintf(strout,"FAIL: MMU: ROM file open: %s.",str);
            RETFAIL(strout);
        }
        #ifdef MMUDEBUG
        logvt->append("MMU: ROM file open.");
        #endif

        ROMsize=GetFileSize(hFile,NULL);
        sprintf(str, "MMU: ROM size: %d bytes.",ROMsize);
        logvt->append(str);
        GameROM=(RAMWORD*)malloc(((ROMsize/4)+1)*sizeof(RAMWORD));
        tempROM=(u8*)malloc(ROMsize);
        if(!GameROM || !tempROM) RETFAIL("FAIL: MMU: GameROM allocation.");
        #ifdef MMUDEBUG
        logvt->append("MMU: GameROM allocated.");
        #endif

        ReadFile(hFile,tempROM,ROMsize,&bytesread,NULL);
        if(bytesread != ROMsize) RETFAIL("FAIL: ROM read.");
        CloseHandle(hFile); hFile=NULL;
        #ifdef MMUDEBUG
        logvt->append("MMU: ROM read and closed.");
        #endif
    }

    for(a=0;a<ROMsize;a++)
    {
        #if WORDS_BIGENDIAN
        GameROM[a>>2].b[3-(a&3)]=tempROM[a];
        #else
        GameROM[a>>2].b[a&3]=tempROM[a];
        EWRAM[4096+(a>>2)].b[a&3]=tempROM[a];
        #endif
    }
    free(tempROM);
    for(a=0;a<(ROMsize>>2);a++)
    {
//    	GameROM[a].op=ARM7opUNL;
//        GameROM[a].cond=ARM7condAL;
        GameROM[a]=ARM7opDecode(GameROM[a].data);
        EWRAM[4096+a]=ARM7opDecode(EWRAM[4096+a].data);
//        GameROM[a].cyc=1;
    }
//    logvt->append(ARM7DASM(GameROM[48].data));

//    fp=fopen("bioshack.bin","rb");
//    logvt->append(BIOSpath);
    hFile=CreateFile(BIOSpath,GENERIC_READ,0,NULL,OPEN_EXISTING,0,NULL);
    if(!hFile) RETFAIL("FAIL: MMU: BIOS file open.");
    BIOSsize=GetFileSize(hFile,NULL);
    BIOS=(RAMWORD*)malloc(BIOSsize/4*sizeof(RAMWORD));
    tempROM=(u8*)malloc(BIOSsize);
    if(!BIOS || !tempROM) RETFAIL("FAIL: MMU: BIOS allocation.");
    ReadFile(hFile,tempROM,BIOSsize,&bytesread,NULL);
    if(bytesread != BIOSsize)
    {
        sprintf(str,"FAIL: BIOS read after %d bytes.",bytesread);
        RETFAIL(str);
    }
    CloseHandle(hFile); hFile=NULL;
    #ifdef MMUDEBUG
    logvt->append("MMU: BIOS read and closed.");
    #endif

    for(a=0;a<BIOSsize;a++)
    {
        #if WORDS_BIGENDIAN
        BIOS[a>>2].b[3-(a&3)]=tempROM[a];
        #else
        BIOS[a>>2].b[a&3]=tempROM[a];
        #endif
    }
    free(tempROM);
    for(a=0;a<BIOSsize/4;a++)
    {
//    	BIOS[a].op=ARM7opUNL;
//        BIOS[a].cond=ARM7condAL;
//        BIOS[a].cyc=1;
        BIOS[a]=ARM7opDecode(BIOS[a].data);
    }

    for(a=0;a<553;a++) VRAMmap[a]=(a&63);

    RETPASS("MMU: Reset.");
}

void MMUfini()
{
    free(EWRAM);
    free(IWRAM);
    free(GameROM);
    logvt->append("MMU: Shutdown.");
}

void MMUremapVRAM(u8 cntindex, u8 vcnt)
{
    u8 master=vcnt&7, offset=(vcnt>>3)&15, a;
    char str[160];
    if(!(vcnt&0x80)) return;
    sprintf(str,"VRAM_CNT_%d: master=%d, offset=%d",cntindex,master,offset);
    switch(cntindex)
    {
        case 0:
        sprintf(str,"%s: phys=00000",str);
        switch(master)  //->00000
        {
            case 0:
                if(!offset) {
                    sprintf(str,"%s, virt=800000",str);
                    for(a=0;a<8;a++) VRAMmap[a+512]=a;
                } break;  //800000
            case 1: switch(offset)
            {
                case 0: sprintf(str,"%s, virt=000000",str); for(a=0;a<8;a++) VRAMmap[a   ]=a; break; //000000
                case 1: sprintf(str,"%s, virt=020000",str); for(a=0;a<8;a++) VRAMmap[a+8 ]=a; break; //020000
                case 2: sprintf(str,"%s, virt=040000",str); for(a=0;a<8;a++) VRAMmap[a+16]=a; break; //040000
                case 3: sprintf(str,"%s, virt=060000",str); for(a=0;a<8;a++) VRAMmap[a+24]=a; break; //060000
                default: break;
            } break;
            case 2: switch(offset)
            {
                case 0: sprintf(str,"%s, virt=400000",str); for(a=0;a<8;a++) VRAMmap[a+256]=a; break; //400000
                case 1: sprintf(str,"%s, virt=420000",str); for(a=0;a<8;a++) VRAMmap[a+264]=a; break; //420000
                default: break;
            } break;
            default: break;
        } break;
        case 1:
        sprintf(str,"%s: phys=20000",str);
        switch(master)  //->20000
        {
            case 0:
                if(!offset) {
                    sprintf(str,"%s, virt=820000",str);
                    for(a=0;a<8;a++) VRAMmap[a+520]=a+8;
                } break;  //820000
            case 1: switch(offset)
            {
                case 0: sprintf(str,"%s, virt=000000",str); for(a=0;a<8;a++) VRAMmap[a   ]=a+8; break; //000000
                case 1: sprintf(str,"%s, virt=020000",str); for(a=0;a<8;a++) VRAMmap[a+8 ]=a+8; break; //020000
                case 2: sprintf(str,"%s, virt=040000",str); for(a=0;a<8;a++) VRAMmap[a+16]=a+8; break; //040000
                case 3: sprintf(str,"%s, virt=060000",str); for(a=0;a<8;a++) VRAMmap[a+24]=a+8; break; //060000
                default: break;
            } break;
            case 2: switch(offset)
            {
                case 0: sprintf(str,"%s, virt=400000",str); for(a=0;a<8;a++) VRAMmap[a+256]=a+8; break; //400000
                case 1: sprintf(str,"%s, virt=420000",str); for(a=0;a<8;a++) VRAMmap[a+264]=a+8; break; //420000
                default: break;
            } break;
            default: break;
        } break;
        case 2:
        sprintf(str,"%s: phys=40000",str);
        switch(master)  //->40000
        {
            case 0:
                if(!offset) {
                    sprintf(str,"%s, virt=840000",str);
                    for(a=0;a<8;a++) VRAMmap[a+528]=a+16;
                } break;  //840000
            case 1: switch(offset)
            {
                case 0: sprintf(str,"%s, virt=000000",str); for(a=0;a<8;a++) VRAMmap[a   ]=a+16; break; //000000
                case 1: sprintf(str,"%s, virt=020000",str); for(a=0;a<8;a++) VRAMmap[a+8 ]=a+16; break; //020000
                case 2: sprintf(str,"%s, virt=040000",str); for(a=0;a<8;a++) VRAMmap[a+16]=a+16; break; //040000
                case 3: sprintf(str,"%s, virt=060000",str); for(a=0;a<8;a++) VRAMmap[a+24]=a+16; break; //060000
                default: break;
            } break;
            case 4:
                if(!offset) {
                    sprintf(str,"%s, virt=200000",str);
                    for(a=0;a<8;a++) VRAMmap[a+128]=a+16;
                } break;  //200000
            default: break;
        } break;
        case 3:
        sprintf(str,"%s: phys=60000",str);
        switch(master)  //->60000
        {
            case 0:
                if(!offset) {
                    sprintf(str,"%s, virt=860000",str);
                    for(a=0;a<8;a++) VRAMmap[a+536]=a+24;
                } break;  //860000
            case 1: switch(offset)
            {
                case 0: sprintf(str,"%s, virt=000000",str); for(a=0;a<8;a++) VRAMmap[a   ]=a+24; break; //000000
                case 1: sprintf(str,"%s, virt=020000",str); for(a=0;a<8;a++) VRAMmap[a+8 ]=a+24; break; //020000
                case 2: sprintf(str,"%s, virt=040000",str); for(a=0;a<8;a++) VRAMmap[a+16]=a+24; break; //040000
                case 3: sprintf(str,"%s, virt=060000",str); for(a=0;a<8;a++) VRAMmap[a+24]=a+24; break; //060000
                default: break;
            } break;
            case 4:
                if(!offset) {
                    sprintf(str,"%s, virt=600000",str);
                    for(a=0;a<8;a++) VRAMmap[a+384]=a+24;
                } break;  //600000
            default: break;
        } break;
        default: break;
    }
    logvt->append(str);
}
/*
#if WORDS_BIGENDIAN

#define MMUintRdB(ram,addr) b=ram[addr].b[3-(addr&3)]
#define MMUintRdH(ram,addr) \
    h=(addr&2)? \
      ((ram[addr].b[0])| \
       (ram[addr].b[1]<<8)): \
      ((ram[addr].b[2])| \
       (ram[addr].b[3]<<8))
#define MMUintRdW(ram,addr) \
    w=(ram[addr].b[0])| \
      (ram[addr].b[1]<<8)| \
      (ram[addr].b[2]<<16)| \
      (ram[addr].b[3]<<24)

#else
*/
#define MMUintRdB(ram,addr) b=ram[(addr)>>2].b[(addr)&3]
#define MMUintRdH(ram,addr) \
    h=((addr)&1) \
        ?(((addr)&2) \
          ?(ram[(addr)>>2].b[3] | ((ram[((addr)>>2)+1].b[0])<<8)) \
          :(ram[(addr)>>2].b[1] | ((ram[(addr)>>2].b[2])<<8))) \
        :(ram[(addr)>>2].h[((addr)&2)>>1])
#define MMUintRdW(ram,addr) \
    switch((addr)&3) \
    { \
/*        case 0: w=ram[(addr)>>2].data; break; \
        case 1: w=(ram[(addr)>>2].b[1]) | ((ram[(addr)>>2].h[1])<<8) | ((ram[((addr)>>2)+1].b[0])<<24); break; \
        case 2: w=(ram[(addr)>>2].h[1]) | ((ram[((addr)>>2)+1].h[0])<<16); break; \
        case 3: w=(ram[(addr)>>2].b[3]) | ((ram[((addr)>>2)+1].h[0])<<8) | ((ram[((addr)>>2)+1].b[2])<<24); break; */ \
        case 0: w=ram[(addr)>>2].data; break; \
        case 1: w=(ram[(addr)>>2].data>> 8)+((ram[(addr)>>2].data&0x000000FF)<<24); break; \
        case 2: w=ram[(addr)>>2].h[1]+((ram[(addr)>>2].h[0])<<16); break; \
        case 3: w=(ram[(addr)>>2].data>>24)+((ram[(addr)>>2].data&0x00FFFFFF)<< 8); break; \
    }
/*
#endif
*/
u8 MMUrdB(u32 prot, u32 addr)
{
    u8 b; char str[80];
    switch(addr&0x0F000000)
    {
        case 0x00000000:
            if(addr&0x00800000) {
                b=DTCM[(addr&0x3FFF)>>2].b[addr&3];
            } else {
                if(prot) MMUintRdB(BIOS, addr&0x000001FF); else return 0;
            }
            break;
        case 0x02000000:
	    MMUintRdB(EWRAM, addr&0x0003FFFFF);
	    if(((addr&0xFFFFFF00)==0x02000000) ||
               ((addr&0xFFFFFF00)==0x0237F000))
            {
//                sprintf(str,"MMU: EWRAMRd: %08X=%02X",addr,b);
//                logvt->append(str);
            }
	    ARM7addClock(2); break;
	case 0x03000000:
	    MMUintRdB(IWRAM, addr&0x00007FFF);
	    break;
	case 0x04000000:
	    if(prot || DSio[(addr&0x1FFF)>>1].flags&REG_FLAG_R)
                b=DSio[(addr&0x1FFF)>>1].b[addr&1];
            else b=0;
            #ifdef MMUDEBUG
            if((!prot) /*&& mmuioimpl[(addr&0x1FFF)>>1]==1*/)
            {
                sprintf(str,"MMU: IORd: %03X=%02X",addr&0x1FFF,b);
//		logvt->append(str);
            }
            #endif
	    break;
	case 0x05000000:
            b=GPUPAL8[addr&0x7FF];
            break;
	case 0x06000000:
	    b=(addr&1)?VRAM[((VRAMmap[(addr>>14)&1023]<<14)+(addr&0x003FFF))>>1]>>8
                      :VRAM[((VRAMmap[(addr>>14)&1023]<<14)+(addr&0x003FFF))>>1]&255;
	    break;
	case 0x07000000:
	    b=(addr&1)?OAM[(addr&0x7FF)>>1]>>8:OAM[(addr&0x7FF)>>1]&255;
	    #ifdef MMUDEBUG
            sprintf(str,"MMU: OAMRd: %03X=%02X",addr&0x7FF,b);
            logvt->append(str);
            #endif
	    break;
	case 0x08000000: case 0x09000000:
	    if((addr&0x00FFFFFF)<=ROMsize)
                MMUintRdB(GameROM, addr&0x00FFFFFF);
            else b=mmuEmpty.b[addr&3];
	    ARM7addClock(4); break;
	case 0x0A000000: case 0x0B000000:
	    if((addr&0x00FFFFFF)<=ROMsize)
                MMUintRdB(GameROM, addr&0x00FFFFFF);
            else b=mmuEmpty.b[addr&3];
	    ARM7addClock(5); break;
	case 0x0C000000: case 0x0D000000:
	    if((addr&0x00FFFFFF)<=ROMsize)
                MMUintRdB(GameROM, addr&0x00FFFFFF);
            else b=mmuEmpty.b[addr&3];
	    ARM7addClock(6); break;
	default: break;
    }
    return b;
}

u16 MMUrdH(u32 prot, u32 addr)
{
    u16 h; char str[80];
    switch(addr&0x0F000000)
    {
        case 0x00000000:
            if(addr&0x00800000) {
                h=DTCM[(addr&0x3FFF)>>2].h[(addr&2)>>1];
            } else {
                if(prot) MMUintRdH(BIOS, addr&0x000001FF); else return 0;
            }
            break;
        case 0x02000000:
	    MMUintRdH(EWRAM, addr&0x0003FFFFF);
	    if(((addr&0xFFFFFF00)==0x02000000) ||
               ((addr&0xFFFFFF00)==0x0237F000))
            {
//                sprintf(str,"MMU: EWRAMRd: %08X=%04X",addr,h);
//                logvt->append(str);
            }
	    ARM7addClock(2); break;
	case 0x03000000:
	    MMUintRdH(IWRAM, addr&0x00007FFF);
	    break;
	case 0x04000000:
	    if(prot || DSio[(addr&0x1FFF)>>1].flags&REG_FLAG_R)
                h=DSio[(addr&0x1FFF)>>1].data;
            else h=0;
            #ifdef MMUDEBUG
            if((!prot)/* && mmuioimpl[(addr&0x1FFF)>>1]==1*/)
            {
                sprintf(str,"MMU: IORd: %03X=%04X",addr&0x1FFF,h);
//		logvt->append(str);
            }
            #endif
	    break;
	case 0x05000000:
	    h=GPUPAL[(addr>>1)&0x3FF];
            break;
	case 0x06000000:
	    h=VRAM[((VRAMmap[(addr>>14)&1023]<<14)+(addr&0x003FFF))>>1];
	    break;
	case 0x07000000:
	    h=OAM[(addr&0x7FF)>>1];
	    #ifdef MMUDEBUG
            sprintf(str,"MMU: OAMRd: %03X=%04X",addr&0x7FF,h);
            logvt->append(str);
            #endif
	    break;
	case 0x08000000: case 0x09000000:
	    if((addr&0x00FFFFFF)<=ROMsize)
                MMUintRdH(GameROM, addr&0x00FFFFFF);
            else h=mmuEmpty.h[(addr&2)>>1];
	    ARM7addClock(4); break;
	case 0x0A000000: case 0x0B000000:
	    if((addr&0x00FFFFFF)<=ROMsize)
                MMUintRdH(GameROM, addr&0x00FFFFFF);
            else h=mmuEmpty.h[(addr&2)>>1];
	    ARM7addClock(5); break;
	case 0x0C000000: case 0x0D000000:
	    if((addr&0x00FFFFFF)<=ROMsize)
                MMUintRdH(GameROM, addr&0x00FFFFFF);
            else h=mmuEmpty.h[(addr&2)>>1];
	    ARM7addClock(6); break;
	default: break;
    }
    return h;
}

u32 MMUrdW(u32 prot, u32 addr)
{
    u32 w; u16 hl,hh; char str[80];
    switch(addr&0x0F000000)
    {
        case 0x00000000:
            if(addr&0x00800000) {
                w=DTCM[(addr&0x3FFF)>>2].data;
            } else {
                if(prot) {MMUintRdW(BIOS, addr&0x000001FF);} else return 0;
            }
            break;
        case 0x02000000:
	    MMUintRdW(EWRAM, addr&0x0003FFFFF);
	    if(((addr&0xFFFFFF00)==0x02000000) ||
               ((addr&0xFFFFFF00)==0x0237F000))
            {
//                sprintf(str,"MMU: EWRAMRd: %08X=%08X",addr,w);
//                logvt->append(str);
            }
	    ARM7addClock(5); break;
	case 0x03000000:
	    MMUintRdW(IWRAM, addr&0x00007FFF);
	    break;
	case 0x04000000:
	    if(prot || DSio[(addr&0x1FFF)>>1].flags&REG_FLAG_R)
                hl=DSio[(addr&0x1FFF)>>1].data;
            else hl=0;
	    addr+=2;
	    if(prot || DSio[(addr&0x1FFF)>>1].flags&REG_FLAG_R)
                hh=DSio[(addr&0x1FFF)>>1].data;
            else hh=0;
            w=hl+(hh<<16);
            #ifdef MMUDEBUG
            if((!prot)/* && mmuioimpl[(addr&0x1FFF)>>1]==1*/)
            {
                sprintf(str,"MMU: IORd: %03X=%08X",addr&0x1FFF,w);
//		logvt->append(str);
            }
            #endif
	    break;
	case 0x05000000:
	    hl=GPUPAL[(addr>>1)&0x3FF];
            addr+=2;
	    hh=GPUPAL[(addr>>1)&0x3FF];
            w=hl+(hh<<16);
            break;
	case 0x06000000:
	    hl=VRAM[((VRAMmap[(addr>>14)&1023]<<14)+(addr&0x003FFF))>>1]; addr+=2;
	    hh=VRAM[((VRAMmap[(addr>>14)&1023]<<14)+(addr&0x003FFF))>>1];
            w=hl+(hh<<16);
            break;
	case 0x07000000:
	    hl=OAM[(addr&0x7FF)>>1]; addr+=2;
	    hh=OAM[(addr&0x7FF)>>1];
            w=hl+(hh<<16);
	    #ifdef MMUDEBUG
            sprintf(str,"MMU: OAMRd: %03X=%08X",addr&0x7FF,w);
            logvt->append(str);
            #endif
            break;
	case 0x08000000: case 0x09000000:
//            sprintf(str,"MMU: RdW at %08X",addr); logvt->append(str);
	    if((addr&0x00FFFFFF)<=ROMsize){ MMUintRdW(GameROM, addr&0x00FFFFFF); }
            else w=mmuEmpty.data;
	    ARM7addClock(7); break;
	case 0x0A000000: case 0x0B000000:
	    if((addr&0x00FFFFFF)<=ROMsize){ MMUintRdW(GameROM, addr&0x00FFFFFF); }
            else w=mmuEmpty.data;
	    ARM7addClock(8); break;
	case 0x0C000000: case 0x0D000000:
	    if((addr&0x00FFFFFF)<=ROMsize){ MMUintRdW(GameROM, addr&0x00FFFFFF); }
            else w=mmuEmpty.data;
	    ARM7addClock(9); break;
	default: break;
    }
    return w;
}

RAMWORD MMUrdS(u32 prot, u32 addr)
{
/*    RAMWORD def;
    def.data=0xEC000000;
    switch(addr&0x0F000000)
//    switch((addr>>24)&15)
    {
        case 0x00000000: return BIOS[(addr&0x001FF)>>2]; break;
        case 0x02000000: return EWRAM[(addr&0x3FFFF)>>2]; break;
        case 0x03000000: return IWRAM[(addr&0x07FFF)>>2]; break;
	case 0x08000000:
	case 0x09000000:
	case 0x0A000000:
	case 0x0B000000:
	case 0x0C000000:
        case 0x0D000000:
	    if((addr&0x00FFFFFF)<=ROMsize) return GameROM[(addr&0xFFFFFF)>>2];
            else return mmuEmpty;
            break;
	//default: return def; break;
    }
/*    int bank=(addr>>24)&15;
    if(MMUbanks[bank]!=NULL)
        return MMUbanks[bank][(addr&MMUmask[bank])>>2];*/

    //E and F handling
    if(addr&0x08000000)
    {
        if((addr&0x00FFFFFF)<=ROMsize) return GameROM[(addr&0xFFFFFF)>>2];
        else return mmuEmpty;
    }
    switch(addr&0x0F000000)
    {
        case 0x03000000: return IWRAM[(addr&0x07FFF)>>2];
        case 0x02000000: return EWRAM[(addr&0x3FFFFF)>>2];
        case 0x00000000: return BIOS[(addr&0x001FF)>>2];
        default: return mmuEmpty;
    }
}
/*
#if WORDS_BIGENDIAN

#define MMUintWrB(ram,addr,val) \
    ram[addr].b[3-(addr&3)]=val; \
    ram[addr].op=ARM7opUNL; \
    ram[addr].cond=ARM7condAL

#define MMUintWrH(ram,addr,val) \
    if(addr&2){ \
        ram[addr].b[0]=val&255; \
        ram[addr].b[1]=val>>8; \
    }else{ \
        ram[addr].b[2]=val&255; \
        ram[addr].b[3]=val>>8; \
    } \
    ram[addr].op=ARM7opUNL; \
    ram[addr].cond=ARM7condAL

#define MMUintWrW(ram,addr,val) \
    ram[addr].b[0]=val&255; \
    ram[addr].b[1]=val>>8; \
    ram[addr].b[2]=val>>16; \
    ram[addr].b[3]=val>>24; \
    ram[addr].op=ARM7opUNL; \
    ram[addr].cond=ARM7condAL

#else
*/
#define MMUintWrB(ram,addr,val) \
    ram[(addr)>>2].b[(addr)&3]=val; \
    ram[(addr)>>2].op=ARM7opUNL; \
    ram[(addr)>>2].cond=ARM7condAL

#define MMUintWrH(ram,addr,val) \
    ram[(addr)>>2].h[((addr)&2)>>1]=val; \
    ram[(addr)>>2].op=ARM7opUNL; \
    ram[(addr)>>2].cond=ARM7condAL

#define MMUintWrW(ram,addr,val) \
    ram[(addr)>>2].data=val; \
    ram[(addr)>>2].op=ARM7opUNL; \
    ram[(addr)>>2].cond=ARM7condAL
/*
#endif
*/
void MMUwrB(u32 bus, u32 addr, u8 val)
{
    char str[80];
    switch(addr&0x0F000000)
    {
        case 0x00000000:
            if(addr&0x00800000) DTCM[(addr&0x3FFF)>>2].b[addr&3]=val;
            break;
        case 0x02000000:
	    MMUintWrB(EWRAM, addr&0x0003FFFFF, val);
	    ARM7addClock(2); break;
	case 0x03000000:
	    MMUintWrB(IWRAM, addr&0x00007FFF, val);
	    break;
        case 0x04000000:
            if(DSio[(addr&0x1FFF)>>1].flags&REG_FLAG_W)
            {
                if((addr&0x1FFE)==0x202) IntClear(val);
                else DSio[(addr&0x1FFF)>>1].b[addr&1]=val;
                if((addr&0x1FF0)==0x240) MMUremapVRAM(addr&15,val);
                DMAcheck(DMA_TIMENOW);
            #ifdef MMUDEBUG
                /*if(mmuioimpl[(addr&0x3FF)>>1]==1)*/
                {
                    sprintf(str,"MMU: IOWr: %03X=%02X",addr&0x1FFF,val);
                    logvt->append(str);
                }
            #endif
            }
	default: break;
    }
}

void MMUwrH(u32 bus, u32 addr, u16 val)
{
    char str[80];
    switch(addr&0x0F000000)
    {
        case 0x00000000:
            if(addr&0x00800000) DTCM[(addr&0x3FFF)>>2].h[(addr&2)>>1]=val;
            break;
        case 0x02000000:
	    MMUintWrH(EWRAM, addr&0x0003FFFFF, val);
	    ARM7addClock(2); break;
	case 0x03000000:
	    MMUintWrH(IWRAM, addr&0x00007FFF, val);
	    break;
	case 0x04000000:
            if(DSio[(addr&0x1FFF)>>1].flags&REG_FLAG_W)
            {
                if((addr&0x1FFE)==0x202) IntClear(val);
                else DSio[(addr&0x1FFF)>>1].data=val;
                if((addr&0x1FF0)==0x240)
                {
                    MMUremapVRAM((addr+0)&15,(val>> 0)&255);
                    MMUremapVRAM((addr+1)&15,(val>> 8)&255);
                }
                DMAcheck(DMA_TIMENOW);
            #ifdef MMUDEBUG
                /*if(mmuioimpl[(addr&0x3FF)>>1]==1)*/
                {
                    sprintf(str,"MMU: IOWr: %03X=%04X",addr&0x1FFF,val);
                    logvt->append(str);
                }
            #endif
            }
	    break;
	case 0x05000000:
	    #ifdef MMUDEBUG
	    sprintf(str,"MMU: PalWr: %08X=%04X",addr,val);
	    logvt->append(str);
	    #endif
	    GPUPAL[(addr>>1)&0x3FF]=val;
	    break;
	case 0x06000000:
	    #ifdef MMUDEBUG
            sprintf(str,"MMU: VRAMWr: %08X [%06X=%06X]=%04X",addr,((addr>>14)&1023)<<14,0x800000+(VRAMmap[(addr>>14)&1023]<<14),val);
            logvt->append(str);
            #endif
	    VRAM[((VRAMmap[(addr>>14)&1023]<<14)+(addr&0x003FFF))>>1]=val; break;
	case 0x07000000:
	    #ifdef MMUDEBUG
            sprintf(str,"MMU: OAMWr: %03X=%04X",addr&0x7FF,val);
            logvt->append(str);
            #endif
	    OAM[(addr&0x7FF)>>1]=val;
            break;
	default: break;
    }
}

void MMUwrW(u32 bus, u32 addr, u32 val)
{
    char str[80];
    switch(addr&0x0F000000)
    {
        case 0x00000000:
            if(addr&0x00800000) DTCM[(addr&0x3FFF)>>2].data=val;
            break;
        case 0x02000000:
	    MMUintWrW(EWRAM, addr&0x0003FFFFF, val);
	    ARM7addClock(2); break;
	case 0x03000000:
	    MMUintWrW(IWRAM, addr&0x00007FFF, val);
	    break;
	case 0x04000000:
            if(DSio[(addr&0x1FFF)>>1].flags&REG_FLAG_W)
            {
                if((addr&0x1FFE)==0x202) IntClear(val&65535);
                else DSio[(addr&0x1FFF)>>1].data=val&65535;
                if((addr&0x1FF0)==0x240)
                {
                    MMUremapVRAM((addr+0)&15,(val>> 0)&255);
                    MMUremapVRAM((addr+1)&15,(val>> 8)&255);
                }
            }
            #ifdef MMUDEBUG
            /*if(mmuioimpl[(addr&0x3FF)>>1]==1)*/
            {
                sprintf(str,"MMU: IOWr: %03X=%08X",addr&0x1FFF,val);
                logvt->append(str);
            }
            #endif
	    addr+=2;
            if(DSio[(addr&0x1FFF)>>1].flags&REG_FLAG_W)
            {
                if((addr&0x1FFE)==0x202) IntClear(val>>16);
                else DSio[(addr&0x1FFF)>>1].data=val>>16;
                if((addr&0x1FF0)==0x240)
                {
                    MMUremapVRAM((addr+0)&15,(val>>16)&255);
                    MMUremapVRAM((addr+1)&15,(val>>24)&255);
                }
            }
            DMAcheck(DMA_TIMENOW);
	    break;
	case 0x05000000:
	    sprintf(str,"MMU: PalWr: %08X=%08X",addr,val);
	    logvt->append(str);
	    GPUPAL[(addr>>1)&0x3FF]=val&65535;
	    addr+=2;
	    GPUPAL[(addr>>1)&0x3FF]=val>>16;
	case 0x06000000:
	    #ifdef MMUDEBUG
            sprintf(str,"MMU: VRAMWr: %08X [%06X=%06X]=%08X",addr,((addr>>14)&1023)<<14,0x800000+(VRAMmap[(addr>>14)&1023]<<14),val);
            logvt->append(str);
            #endif
	    VRAM[((VRAMmap[(addr>>14)&1023]<<14)+(addr&0x003FFF))>>1]=val&65535; addr+=2;
	    VRAM[((VRAMmap[(addr>>14)&1023]<<14)+(addr&0x003FFF))>>1]=val>>16;
	    break;
	case 0x07000000:
	    #ifdef MMUDEBUG
            sprintf(str,"MMU: OAMWr: %03X=%08X",addr&0x7FF,val);
            logvt->append(str);
            #endif
	    OAM[(addr&0x7FF)>>1]=val&65535; addr+=2;
	    OAM[(addr&0x7FF)>>1]=val>>16;
	    break;
	default: break;
    }
}

void MMUwrS(u32 bus, u32 addr, RAMWORD val)
{
    switch(addr&0x0F000000)
    {
        case 0x02000000: EWRAM[(addr&0x3FFFFF)>>2]=val; break;
        case 0x03000000: IWRAM[(addr&0x07FFF)>>2]=val; break;
	case 0x08000000:
	case 0x09000000:
	case 0x0A000000:
	case 0x0B000000:
	case 0x0C000000:
        case 0x0D000000: GameROM[(addr&0xFFFFFF)>>2]=val; break;
	default: break;
    }
}

/*** EOF:mmu.c ***********************************************************/

