/**************************************************************************
* DSemu: GameBoy Advance interface and encapsulation (gba.c)              *
* Released under the terms of the BSD Public Licence                      *
* Imran Nazar (tf@oopsilon.com), 2004                                     *
**************************************************************************/

#include <windows.h>
#include <stdio.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glaux.h>
#include "arm7.h"
#include "mmu.h"
#include "gpu.h"
#include "defs.h"
#include "vtbl.h"
#include "win.h"
#include "err.h"
#include "emu.h"
#include "gba.h"
#include "res.h"
#include "key.h"
#include "dsioreg.h"

#define TOUCHIPC 0x02000000

LOGVTBL *logvt=NULL;

IOREG DSio[4096];

u16 ioregflags[]={
    3,3,3,1,3,3,3,3, 2,2,2,2,2,2,2,2, 2,2,2,2,2,2,2,2, 2,2,2,2,2,2,2,2,
    2,2,2,2,3,3,2,0, 3,2,2,0,0,0,0,0, 3,3,3,0,3,0,3,0, 3,3,3,0,3,0,3,0,
    3,3,3,0,0,0,0,0, 3,3,3,3,3,3,3,3, 3,3,3,3,0,0,0,0, 2,2,2,2,2,3,2,2,
    2,2,2,3,2,2,2,2, 2,3,2,2,2,2,2,3, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,

    3,3,3,3,3,3,3,3, 0,0,0,0,0,0,0,0, 3,3,3,3,3,3,0,0, 1,3,3,0,0,0,0,0,
    3,0,0,0,0,0,0,0, 3,3,3,3,1,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,

    3,3,3,0,3,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,
    3,3,3,3,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,

    0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,
};

u32 *screenbuf=NULL, *outbuf=NULL, *dbgbuf=NULL, *palbuf=NULL, *membuf=NULL;
HDC hDCMain, hDCPal, hDCDbg, hDCMem;
BITMAPV4HEADER BmpInfo;
HWND hWnd, hWndPal, hWndDbg, hWndMem;
int stepclk, dbgmode, memmode; u32 dumpaddr;

u16 KeyMap[256]={0};

HGLRC hRC;
int statbarSize;
static GLuint backbufTex;
float winX=256.0, winY=384.0;
int bufferFlipped=0, usingGL=0;

void ResizeWin(HWND wnd, int x, int y)
{
    RECT rWnd, rCli, rDiff, rStat;
    HWND hStatus;
    int neww, newh;
    char str[80];

    GetWindowRect(wnd, &rWnd);
    GetClientRect(wnd, &rCli);

    SetRect(&rDiff,
            0,
            0,
            (rWnd.right - rWnd.left) - rCli.right,
            (rWnd.bottom - rWnd.top) - rCli.bottom);

    hStatus = GetDlgItem(wnd, ID_STATUS);
    if(hStatus)
    {
        SendMessage(hStatus, WM_SIZE, 0, 0);
        GetWindowRect(hStatus, &rStat);
    }
    else SetRect(&rStat,0,0,0,0);

    neww = x + rDiff.right;
    newh = y + rDiff.bottom + (rStat.bottom - rStat.top);

    SetWindowPos(wnd, HWND_TOP, 0, 0, neww, newh, SWP_NOMOVE|SWP_NOZORDER);
}

EMUVTBL *getVt()
{
    return &vtbl;
}

int emuInit(char *file, LOGVTBL *lvtbl, HWND wnd, HWND pwnd, HWND dwnd, HWND mwnd)
{
    char str[40], inifile[MAX_PATH*2]; int a,inipathlen;
    PIXELFORMATDESCRIPTOR pfd; GLuint pf;
    HWND hStat; RECT rStat;

    vtbl.running=0;
    vtbl.animate=0;
    logvt=lvtbl;

    stepclk=0;
    hWnd=wnd; hWndPal=pwnd; hWndDbg=dwnd; hWndMem=mwnd;

    dbgbuf = (u32*)malloc(468*144*2);
    if(!dbgbuf) RETFAIL("FAIL: Main: Buffer init.");
    memset(dbgbuf, 0, 468*144*2);

    if(ARM7init(file,dbgbuf)) RETFAIL("FAIL: Main: ARM7 init.");
    if(GPUinit()) RETFAIL("FAIL: Main: GPU init.");

    ResizeWin(hWndPal, 280, 320);
    ResizeWin(hWndDbg, 480, 160);
    ResizeWin(hWndMem, 360, 160);

    screenbuf = (u32*)malloc(256*384*2);
    if(!screenbuf) RETFAIL("FAIL: Main: Buffer init.");
    memset(screenbuf, 0, 256*384*2);

    outbuf = (u32*)malloc(256*384*2);
    if(!outbuf) RETFAIL("FAIL: Main: Buffer init.");
    memset(outbuf, 0, 256*384*2);

    palbuf = (u32*)malloc(264*312*2);
    if(!palbuf) RETFAIL("FAIL: Main: Buffer init.");
    memset(palbuf, 0, 264*312*2);

    membuf = (u32*)malloc(360*160*2);
    if(!membuf) RETFAIL("FAIL: Main: Buffer init.");
    memset(membuf, 0, 360*160*2);

    logvt->append("Main: Memory allocated.");

    hDCMain = GetDC(hWnd);
    hDCPal  = GetDC(hWndPal);
    hDCDbg  = GetDC(hWndDbg);
    hDCMem  = GetDC(hWndMem);

    memset(&BmpInfo, 0, sizeof(BmpInfo));
    BmpInfo.bV4Size = sizeof(BmpInfo);
    BmpInfo.bV4Planes = 1;           // Must be one, apparently
    BmpInfo.bV4BitCount = 16;
    BmpInfo.bV4V4Compression = BI_RGB|BI_BITFIELDS;
    BmpInfo.bV4RedMask = 0x001F;
    BmpInfo.bV4GreenMask = 0x03E0;
    BmpInfo.bV4BlueMask = 0x7C00;

    inipathlen=strstr(logvt->file,"\\log.txt")-logvt->file;
    strncpy(inifile, logvt->file, inipathlen); inifile[inipathlen]=0;
    sprintf(inifile, "%s\\dsemu.ini", inifile);
    GetPrivateProfileString("Keys","UP","26",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_UP;
    GetPrivateProfileString("Keys","DOWN","28",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_DOWN;
    GetPrivateProfileString("Keys","LEFT","25",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_LEFT;
    GetPrivateProfileString("Keys","RIGHT","27",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_RIGHT;
    GetPrivateProfileString("Keys","A","5A",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_A;
    GetPrivateProfileString("Keys","B","58",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_B;
    GetPrivateProfileString("Keys","L","41",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_L;
    GetPrivateProfileString("Keys","R","53",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_R;
    GetPrivateProfileString("Keys","START","0D",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_START;
    GetPrivateProfileString("Keys","SELECT","08",str,4,inifile);
    KeyMap[strtol(str,NULL,16)&255]=GBAKEY_SELECT;

    GetPrivateProfileString("General","FixFPS","1",str,4,inifile);
    vtbl.fixfps=strtol(str,NULL,10)&1;

    logvt->append("Main: INI file read.");

    hStat = GetDlgItem(wnd, ID_STATUS);
    SendMessage(hStat, WM_SIZE, 0, 0);
    GetWindowRect(hStat, &rStat);
    statbarSize=rStat.bottom-rStat.top;

    //---BEGIN-STEAL-------------------------------------------------------
    memset(&pfd,0,sizeof(pfd));
    pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
    pfd.nVersion = 1;
    pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
    pfd.iPixelType = PFD_TYPE_RGBA;
    pfd.cColorBits = 16;
    pfd.cDepthBits = 16;
    pfd.cAccumBits = 0;
    pfd.cStencilBits = 0;
    if(!(pf=ChoosePixelFormat(hDCMain,&pfd)))
    {
        logvt->append("FAIL: OpenGL: Pixel format resolution.");
        usingGL=0; goto glInitEnd;
    }
//    logvt->append("OpenGL: Pixel format found.");
    SetPixelFormat(hDCMain, pf, &pfd);
//    logvt->append("OpenGL: Pixel format selected.");
    hRC=wglCreateContext(hDCMain);
//    logvt->append("OpenGL: Rendering context created.");
    if(wglMakeCurrent(hDCMain, hRC)==FALSE)
    {
        logvt->append("FAIL: OpenGL: Rendering context selection.");
        usingGL=0; goto glInitEnd;
    }
//    logvt->append("OpenGL: Rendering context selected.");

    glPolygonMode (GL_FRONT_AND_BACK, GL_FILL);

    glDisable(GL_CULL_FACE);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_BLEND);

    glShadeModel(GL_SMOOTH);

    glEnable(GL_TEXTURE_2D);
    glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);

    glGenTextures(1,&backbufTex);

    glBindTexture(GL_TEXTURE_2D,backbufTex);
    glPixelStorei(GL_UNPACK_ALIGNMENT,1);
    glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
    glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

    logvt->append("OpenGL: About to create texture image.");

    glTexImage2D(GL_TEXTURE_2D,0,GL_RGB5_A1,256,512,0, GL_RGBA,0x8034,outbuf);
    if(glGetError()!=GL_NO_ERROR)
    {
        logvt->append("FAIL: OpenGL: Texture image creation.");
        usingGL=0; goto glInitEnd;
    }
    usingGL=1;
    //---END-STEAL---------------------------------------------------------
glInitEnd:
    if(usingGL) logvt->append("GPU: Using OpenGL.");
    else logvt->append("GPU: Using GDI.");

    ResizeWin(hWnd, 256, 384);

    dbgmode=3; memmode=1;
    dumpaddr=0x04000000;
    for(a=0;a<4096;a++) DSio[a].flags=0;
    for(a=0;a<512;a++)
    {
        DSio[a].flags=ioregflags[a];
        DSio[a+2048].flags=ioregflags[a];
    }
    if(emuReset()) RETFAIL("FAIL: Main: Initialisation.");
    RETPASS("Main: Emulator initialised.");
}

void gbaGLblit()
{
    __int64 maskr=0x001F001F001F001F;
    __int64 maskg=0x03E003E003E003E0;
    __int64 maskb=0x7C007C007C007C00;
    float ypos=(winY)/((winY)+(float)(statbarSize*2.0))-0.005; //<3 magic numbers

    if(!bufferFlipped)
    {
        __asm {
        push esi
        push edi
        mov esi,screenbuf
        mov edi,outbuf
        mov ecx,256*384/4
fliplp:
        movq mm0,[esi]
        movq mm1,mm0
        movq mm2,mm0
        pand mm0,[maskr]
        pand mm1,[maskg]
        pand mm2,[maskb]
        psllw mm0,11
        psllw mm1,1
        psrlw mm2,9
        por mm0,mm1
        por mm0,mm2
        movq [edi],mm0
        add esi,8
        add edi,8
        dec ecx
        jnz fliplp
        emms
        pop edi
        pop esi
        }
        bufferFlipped=1;
    }

	glClearColor(0,0,0,0);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glBindTexture(GL_TEXTURE_2D,backbufTex);
	glTexSubImage2D(GL_TEXTURE_2D,0,0,0,256,384, GL_RGBA, 0x8034, outbuf);
        if(glGetError()!=GL_NO_ERROR) logvt->append("FAIL: OpenGL: Texture image selection.");

		glEnable(GL_TEXTURE_2D);
		glBegin(GL_QUADS);
			glTexCoord2f(1,0);
			glVertex2f(1,1);
			glTexCoord2f(0,0);
			glVertex2f(-1,1);
			glTexCoord2f(0,0.75f);
			glVertex2f(-1,-ypos);
			glTexCoord2f(1,0.75f);
			glVertex2f(1,-ypos);
		glEnd();
		glDisable(GL_TEXTURE_2D);

	SwapBuffers(hDCMain);
}

void emuFini()
{
    logvt->append("Main: Emulator shutting down.");
    GPUfini();
    ARM7fini();

    glDeleteTextures(1,&backbufTex);
    if(hRC)
    {
        wglMakeCurrent(NULL,NULL);
        wglDeleteContext(hRC); hRC=NULL;
    }

    memset(screenbuf, 0, 256*384*2);
    memset(dbgbuf, 0, 468*144*2);
    memset(palbuf, 0, 128*144*2);
    memset(membuf, 0, 360*160*2);

    if(usingGL) gbaGLblit();
    else {
    BmpInfo.bV4Width = 256;
    BmpInfo.bV4Height = -384;
    SetDIBitsToDevice(hDCMain, 0, 0, 256, 384, 0, 0, 0, 384,
                      screenbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
    }

    BmpInfo.bV4Width = 468;
    BmpInfo.bV4Height = -144;
    SetDIBitsToDevice(hDCDbg, 6, 8, 468, 144, 0, 0, 0, 144,
                      dbgbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    BmpInfo.bV4Width = 264;
    BmpInfo.bV4Height = -312;
    SetDIBitsToDevice(hDCPal, 8, 8, 264, 312, 0, 0, 0, 312,
                      palbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    BmpInfo.bV4Width = 360;
    BmpInfo.bV4Height = -160;
    SetDIBitsToDevice(hDCMem, 6, 0, 360, 160, 0, 0, 0, 160,
                      membuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    ReleaseDC(hWnd, hDCMain);
    ReleaseDC(hWndPal, hDCPal);
    ReleaseDC(hWndDbg, hDCDbg);
    ReleaseDC(hWndMem, hDCMem);
    hWnd=NULL;    hWndPal=NULL; hWndDbg=NULL; hWndMem=NULL;
    hDCMain=NULL; hDCPal=NULL;  hDCDbg=NULL;  hDCMem=NULL;

    if(membuf) free(membuf);
    if(dbgbuf) free(dbgbuf);
    if(palbuf) free(palbuf);
    if(screenbuf) free(screenbuf);
    membuf=NULL; dbgbuf=NULL; palbuf=NULL; screenbuf=NULL;
}

void emuRefresh()
{
//    logvt->append("Main: If we fail, we shouldn't be here.");
    GPUpal((u16*)palbuf);

    if(usingGL) gbaGLblit();
    else {
    BmpInfo.bV4Width = 256;
    BmpInfo.bV4Height = -384;
    SetDIBitsToDevice(hDCMain, 0, 0, 256, 384, 0, 0, 0, 384,
                      screenbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
    }

    BmpInfo.bV4Width = 468;
    BmpInfo.bV4Height = -144;
    SetDIBitsToDevice(hDCDbg, 6, 8, 468, 144, 0, 0, 0, 144,
                      dbgbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    BmpInfo.bV4Width = 264;
    BmpInfo.bV4Height = -312;
    SetDIBitsToDevice(hDCPal, 8, 8, 264, 312, 0, 0, 0, 312,
                      palbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    BmpInfo.bV4Width = 360;
    BmpInfo.bV4Height = -160;
    SetDIBitsToDevice(hDCMem, 6, 0, 360, 160, 0, 0, 0, 160,
                      membuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
}

void emuStep()
{
    static int stepwhere=1;
    stepclk+=ARM7exec();
    ARM7status(0,dbgmode);
    emuMMUDump(dumpaddr,memmode);
    SendMessage(hWndDbg, WM_OFFRESET, 0, 0);
    if(stepclk>=1536 && stepwhere==1)
    {
        stepclk=0; stepwhere=2;
//        logvt->append("Line!");
        if(vtbl.gpuon) GPUscanline((u16*)screenbuf);
    }
    if(stepclk>=594 && stepwhere==2)
    {
        stepclk=0; stepwhere=1;
        if(vtbl.gpuon) GPUclearHBL();
    }
    emuRefresh();
}

int emuReset()
{
    int a;
    for(a=0;a<4096;a++) DSio[a].data=0;
    DSio[REG_KEYINPUT].data=0x03FF;

    if(ARM7reset()) RETFAIL("FAIL: ARM7: Reset.");
    GPUreset();

    MMUwrH(0,TOUCHIPC+20,0xFFFF);
    memset(screenbuf, 0, 256*384*2);
    memset(dbgbuf, 0, 468*144*2);
    memset(palbuf, 0, 128*144*2);
    memset(membuf, 0, 360*160*2);
    SendMessage(hWndDbg, WM_OFFRESET, 0, 0);

    ARM7status(0,dbgmode);
    emuMMUDump(dumpaddr,memmode);

    if(usingGL) { bufferFlipped=0; gbaGLblit(); }
    else {
    BmpInfo.bV4Width = 256;
    BmpInfo.bV4Height = -384;
    SetDIBitsToDevice(hDCMain, 0, 0, 256, 384, 0, 0, 0, 384,
                      screenbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
    }

    BmpInfo.bV4Width = 468;
    BmpInfo.bV4Height = -144;
    SetDIBitsToDevice(hDCDbg, 6, 8, 468, 144, 0, 0, 0, 144,
                      dbgbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    BmpInfo.bV4Width = 264;
    BmpInfo.bV4Height = -312;
    SetDIBitsToDevice(hDCPal, 8, 8, 264, 312, 0, 0, 0, 312,
                      palbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);

    BmpInfo.bV4Width = 360;
    BmpInfo.bV4Height = -160;
    SetDIBitsToDevice(hDCMem, 6, 0, 360, 160, 0, 0, 0, 160,
                      membuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
    return 0;
}

void emuFrame()
{
    int a;
    for(a=0;a<263;a++)
    {
        if(ARM7execfor(1536)) break;
        if(vtbl.gpuon) GPUscanline((u16*)screenbuf);
        if(ARM7execfor(594)) break;
        if(vtbl.gpuon) GPUclearHBL();
    }
    if(usingGL) { bufferFlipped=0; gbaGLblit(); }
    else {
    BmpInfo.bV4Width = 256;
    BmpInfo.bV4Height = -384;
    SetDIBitsToDevice(hDCMain, 0, 0, 256, 384, 0, 0, 0, 384,
                      screenbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
    }
}

void emuLine()
{
    ARM7execfor(1536);
    if(vtbl.gpuon) GPUscanline((u16*)screenbuf);
    ARM7execfor(594);
    if(vtbl.gpuon) GPUclearHBL();
    if(usingGL) { bufferFlipped=0; gbaGLblit(); }
    else {
    BmpInfo.bV4Width = 256;
    BmpInfo.bV4Height = -384;
    SetDIBitsToDevice(hDCMain, 0, 0, 256, 384, 0, 0, 0, 384,
                      screenbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
    }
}

void emuLineDbg()
{
    int a;
    ARM7execfor(1536);
    if(vtbl.gpuon) GPUscanline((u16*)screenbuf);
    ARM7execfor(594);
    if(vtbl.gpuon) GPUclearHBL();
    ARM7status(0,dbgmode);
    emuMMUDump(dumpaddr,memmode);
    SendMessage(hWndDbg, WM_OFFRESET, 0, 0);
    bufferFlipped=0; emuRefresh();
}

void emuPause()
{
    SendMessage(hWnd, WM_COMMAND, ID_MENU_DBG_PAUSE, 0);
    SendMessage(hWndDbg, WM_PAINT, 0, 0);
}

void emuResize(int lParam)
{
    char str[160];
    winX=(float)LOWORD(lParam);
    winY=(float)(HIWORD(lParam)-statbarSize);
//    sprintf(str,"Main: Window resized to %dx%d.",LOWORD(lParam),HIWORD(lParam));
//    logvt->append(str);
    if(usingGL) glViewport(0,0,LOWORD(lParam),HIWORD(lParam));
}

void emuDebugCPU(int off,int mode)
{
    dbgmode=mode;
    ARM7status(off,dbgmode);
}

#define dump16b(p) \
    for(a=0;a<16;a++) \
    { \
        sprintf(str, \
	    "%08X: %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X\n", \
            (p)+a*16, \
	    MMUrdB(1,(p)+a*16+0x0),MMUrdB(1,(p)+a*16+0x1), \
	    MMUrdB(1,(p)+a*16+0x2),MMUrdB(1,(p)+a*16+0x3), \
	    MMUrdB(1,(p)+a*16+0x4),MMUrdB(1,(p)+a*16+0x5), \
	    MMUrdB(1,(p)+a*16+0x6),MMUrdB(1,(p)+a*16+0x7), \
	    MMUrdB(1,(p)+a*16+0x8),MMUrdB(1,(p)+a*16+0x9), \
	    MMUrdB(1,(p)+a*16+0xA),MMUrdB(1,(p)+a*16+0xB), \
	    MMUrdB(1,(p)+a*16+0xC),MMUrdB(1,(p)+a*16+0xD), \
	    MMUrdB(1,(p)+a*16+0xE),MMUrdB(1,(p)+a*16+0xF)); \
        dbgOut(membuf,360,160,str,0,24+a*8,0x7FFF); \
    }

#define dump16h(p) \
    (p)&=0xFFFFFFF0; \
    for(a=0;a<16;a++) \
    { \
        sprintf(str, \
	    "%08X: %04X %04X %04X %04X %04X %04X %04X %04X\n", \
            (p)+a*16, \
	    MMUrdH(1,(p)+a*16+0x0),MMUrdH(1,(p)+a*16+0x2), \
	    MMUrdH(1,(p)+a*16+0x4),MMUrdH(1,(p)+a*16+0x6), \
	    MMUrdH(1,(p)+a*16+0x8),MMUrdH(1,(p)+a*16+0xA), \
	    MMUrdH(1,(p)+a*16+0xC),MMUrdH(1,(p)+a*16+0xE)); \
        dbgOut(membuf,360,160,str,0,24+a*8,0x7FFF); \
    }

#define dump16w(p) \
    for(a=0;a<16;a++) \
    { \
        sprintf(str,"%08X: ",(p)+a*16); \
        dbgOut(membuf,360,160,str,0,24+a*8,0x7FFF); \
        w=(p)+a*16+0x0; sprintf(str,"%08X",MMUrdW(1,w)); \
        if(arm7reg.r[13]==w) dbgOut(membuf,360,160,str,60,24+a*8,0x03FF); \
        else                 dbgOut(membuf,360,160,str,60,24+a*8,0x7FFF); \
        w=(p)+a*16+0x4; sprintf(str,"%08X",MMUrdW(1,w)); \
        if(arm7reg.r[13]==w) dbgOut(membuf,360,160,str,114,24+a*8,0x03FF); \
        else                 dbgOut(membuf,360,160,str,114,24+a*8,0x7FFF); \
        w=(p)+a*16+0x8; sprintf(str,"%08X",MMUrdW(1,w)); \
        if(arm7reg.r[13]==w) dbgOut(membuf,360,160,str,168,24+a*8,0x03FF); \
        else                 dbgOut(membuf,360,160,str,168,24+a*8,0x7FFF); \
        w=(p)+a*16+0xC; sprintf(str,"%08X",MMUrdW(1,w)); \
        if(arm7reg.r[13]==w) dbgOut(membuf,360,160,str,222,24+a*8,0x03FF); \
        else                 dbgOut(membuf,360,160,str,222,24+a*8,0x7FFF); \
    }

void emuMMUDump(u32 addr, int mode)
{
    int a; u32 w; char str[512];
    dumpaddr=addr;
    dbgOutClear(membuf,360,160);
    memmode=mode;
    switch(mode)
    {
        case 1: dump16b(addr); break;
        case 2: dump16h(addr); break;
        case 3: dump16w(addr); break;
    }
}

void emuGPUCol(int idx)
{
    GPUpalcol((u16*)palbuf,idx);
    BmpInfo.bV4Width = 264;
    BmpInfo.bV4Height = -312;
    SetDIBitsToDevice(hDCPal, 8, 8, 264, 312, 0, 0, 0, 312,
                      palbuf, (BITMAPINFO*)&BmpInfo, DIB_RGB_COLORS);
}

void emuKeySet(u32 vk, u32 updown)
{
    int touchx, touchy; u16 t;
    char str[80];
    if(vk==VK_LBUTTON)
    {
        touchx=(LOWORD(updown)&0xFFFF)*14;
        touchy=(((HIWORD(updown)&0x7FFF)-192)*8917)>>9;
        MMUwrH(0,TOUCHIPC+4, touchx);
        MMUwrH(0,TOUCHIPC+6, touchy);
        t=MMUrdH(0,TOUCHIPC+20); t&=0xFFBF;
        if(!(updown&0x80000000)) t|=0x40;
        MMUwrH(0,TOUCHIPC+20,t);
/*        sprintf(str,"WND: Touch: %d,%d - %d",LOWORD(updown),
                                             (HIWORD(updown)&0x7FFF),
                                             ((updown&0x80000000)?1:0));
        logvt->append(str);
*/
    } else {
        if(updown==1)
            DSio[REG_KEYINPUT].data&=(0xFFFF-KeyMap[vk&255]);
        else if(updown==0)
            DSio[REG_KEYINPUT].data|=(KeyMap[vk&255]);
    }
}

/*** EOF:gba.c ***********************************************************/
